//$Id$
//@author Cory Sharp <cssharp@eecs.berkeley.edu>

module GlowRadioM
{
  provides interface StdControl;
  uses interface TimerMilli;
  uses interface LedsIntensity;
  uses interface SendMsg;
  uses interface ReceiveMsg;
}
implementation
{
  int8_t m_acc[3];
  uint8_t m_state[3];

  enum {
    MAX_VAL = 32,
    MAX_DELAY_COUNT = 8,

    LED_OFF = 0,
    LED_UP = 1,
    LED_DOWN = 2,

    NO_LED = 3,
    CYCLE_END = 4,
  };

  const int8_t m_cycle[] = {
    0, 1, 2, NO_LED, NO_LED, NO_LED, NO_LED, NO_LED, NO_LED,
    2, 1, 0, NO_LED, NO_LED, NO_LED, NO_LED, NO_LED, NO_LED,
    CYCLE_END,
  };

  uint8_t m_delay_count;
  uint8_t m_cycle_count;

  typedef struct
  {
    uint8_t ledNum;
  } GlowMsg_t;

  TOS_Msg m_msg;
  bool m_is_sending;

  command result_t StdControl.init()
  {
    m_acc[0] = 0;
    m_acc[1] = 0;
    m_acc[2] = 0;
    m_state[0] = LED_OFF;
    m_state[1] = LED_OFF;
    m_state[2] = LED_OFF;
    m_delay_count = 0;
    m_cycle_count = 0;
    m_is_sending = FALSE;
    return SUCCESS;
  }

  command result_t StdControl.start()
  {
    call TimerMilli.setPeriodic( 20 );
    return SUCCESS;
  }

  command result_t StdControl.stop()
  {
    call TimerMilli.stop();
    return SUCCESS;
  }

  void incLeds()
  {
    int i;
    for( i=0; i<3; i++ )
    {
      switch( m_state[i] )
      {
	case LED_OFF:
	  break;
	case LED_UP:
	  if( ++m_acc[i] >= (MAX_VAL-1) )
	  {
	    m_acc[i] = (MAX_VAL-1);
	    m_state[i] = LED_DOWN;
	  }
	  call LedsIntensity.set( i, m_acc[i] << 3 );
	  break;
	case LED_DOWN:
	  if( --m_acc[i] <= 0 )
	  {
	    m_acc[i] = 0;
	    m_state[i] = LED_OFF;
	  }
	  call LedsIntensity.set( i, m_acc[i] << 3 );
	  break;
      }
    }
  }

  void cycleLeds()
  {
    if( ++m_delay_count >= MAX_DELAY_COUNT )
    {
      m_delay_count = 0;

      if( m_cycle[m_cycle_count] < NO_LED )
      {
	uint8_t ledNum = m_cycle[m_cycle_count];
	m_state[ledNum] = LED_UP;
	if( m_is_sending == FALSE )
	{
	  GlowMsg_t* body = (GlowMsg_t*)m_msg.data;
	  body->ledNum = ledNum;
	  if( call SendMsg.send( TOS_BCAST_ADDR, sizeof(GlowMsg_t), &m_msg ) == SUCCESS )
	    m_is_sending = TRUE;
	}
      }

      if( m_cycle[++m_cycle_count] == CYCLE_END )
	m_cycle_count = 0;
    }
  }

  event result_t TimerMilli.fired()
  {
    if( TOS_LOCAL_ADDRESS == 1 )
      cycleLeds();
    incLeds();
    return SUCCESS;
  }

  event result_t SendMsg.sendDone( TOS_MsgPtr msg, result_t success )
  {
    m_is_sending = FALSE;
    return SUCCESS;
  }

  event TOS_MsgPtr ReceiveMsg.receive( TOS_MsgPtr msg )
  {
    if( TOS_LOCAL_ADDRESS != 1 )
    {
      GlowMsg_t* body = (GlowMsg_t*)msg->data;
      if( body->ledNum < NO_LED )
	m_state[body->ledNum] = LED_UP;
    }
    return msg;
  }
}

