/**
 * Large-scale Reliable Transfer (LRX) Receive
 * 
 * Transfer large amount of data reliably.
 *
 * Trnasfer an array of data blocks, which is called data cluster. Each data
 * block fits into one packet.
 * To start transfer, description of data should be given. Receiver can deny
 * data after looking at the description of data. (for example, receiver
 * already has that data)
 *
 * User of LRXSend should provide buffering. When LRXSend asks for
 * a specific block, user should fill up buffer of LRXSend. When LRXReceive
 * gives a pointer to buffer of LRXReceive, user of LRXReceive should copy
 * content to its own buffer, so that LRXReceive can receive next packet
 * into the buffer of LRXReceive.
 *
 * @author Sukun Kim
 * @author Philip Buonadonna
 */

interface LRXReceive {
	/**
	 * Transfer is requested. Decide whether to receive <code>numofBlock</code>
	 * blocks from source <code>sourceID</code> with
	 * description(<code>desc</code>) of size(<code>descSize</code>).
	 *
	 * @return SUCCESS if upper layer accepts receive, FAIL otherwise.
	 */
	event result_t transferRequested(uint16_t sourceID, uint8_t numofBlock,
		uint8_t *desc, uint8_t descSize);
	
	/**
	 * Receive completed. <code>success</code> indicates whether the receive
	 * was successful or not.
	 *
	 * @return SUCCESS always.
	 */
	event result_t acceptedTransferDone(result_t success);
	
	/**
	 * Abort receiving.
	 *
	 * @return SUCCESS always.
	 */
	command result_t abortReceive();

	/**
	 * A data block <code>blockNum</code> is received. Its size is
	 * <code>blockSize</code> and its content is in <code>blockBuf</code>.
	 *
	 * @return SUCCESS always.
	 */
	event result_t writeDataBlock(uint8_t blockNum, uint8_t *blockBuf,
		uint8_t blockSize);
}

