//
// module for the TC77 temp chip
//
//

module TC77M {
    provides {
	interface ADC;
	interface StdControl;
    } uses {
	interface Timer;
    }
}
implementation {
    norace uint8_t flag;
    uint16_t data;
    
    // put the tc77 in shutdown mode until a 'post start'
    // is called
    task void init() {
	uint8_t i;
	flag = 1;
	
	TOSH_MAKE_TC77_SCK_OUTPUT();
	TOSH_MAKE_TC77_CS_OUTPUT();

	// drop the CS pin to select TC77
	TOSH_CLR_TC77_CS_PIN();

	// cycle the clock 16 times to dummy a temperature read
	TOSH_MAKE_TC77_SIO_INPUT();
	for (i = 0; i < 16; i++) {
	    TOSH_uwait(5);
	    TOSH_SET_TC77_SCK_PIN();
	    TOSH_uwait(5);
	    TOSH_CLR_TC77_SCK_PIN();
	}
	
	// cycle the clock 16 times to shift out 0xffff (power-down)
	TOSH_MAKE_TC77_SIO_OUTPUT();	
	TOSH_SET_TC77_SIO_PIN();
	for (i = 0; i < 16; i++) {
	    TOSH_uwait(5);
	    TOSH_SET_TC77_SCK_PIN();
	    TOSH_uwait(5);
	    TOSH_CLR_TC77_SCK_PIN();
	}

	// raise the CS pin to deselect TC77
	TOSH_SET_TC77_CS_PIN();
	
	flag = 0;
    }

    // wake up the TC77 sensor and call a one-shot
    // timer to wake up in 300ms
    task void start() {
	uint8_t i;

	// drop the CS pin to select TC77
	TOSH_CLR_TC77_CS_PIN();

	// cycle the clock 16 times to dummy a temperature read
	TOSH_MAKE_TC77_SIO_INPUT();
	for (i = 0; i < 16; i++) {
	    TOSH_uwait(5);
	    TOSH_SET_TC77_SCK_PIN();
	    TOSH_uwait(5);
	    TOSH_CLR_TC77_SCK_PIN();
	}
	
	// cycle the clock 16 times to shift out 0x0000 (power-up)
	TOSH_MAKE_TC77_SIO_OUTPUT();	
	TOSH_CLR_TC77_SIO_PIN();
	for (i = 0; i < 16; i++) {
	    TOSH_uwait(5);
	    TOSH_SET_TC77_SCK_PIN();
	    TOSH_uwait(5);
	    TOSH_CLR_TC77_SCK_PIN();
	}

	// raise the CS pin to deselect TC77
	TOSH_SET_TC77_CS_PIN();
	
	if (call Timer.start(TIMER_ONE_SHOT, 400) == SUCCESS)
	    return;

	// if timer start failed, power down the tc77
	post init();
    }
    
    // after the one-shot timer has woken us up again,
    // get a reading off of the TC77 and shut it down.
    task void finish() {
	uint8_t i;
	data = 0;

	// drop the CS pin to select TC77
	TOSH_CLR_TC77_CS_PIN();

	// cycle the clock 16 times to shift in temperature bits
	TOSH_MAKE_TC77_SIO_INPUT();
	for (i = 0; i < 16; i++) {
	    TOSH_uwait(5);
	    TOSH_SET_TC77_SCK_PIN();
	    TOSH_uwait(5);
	    data |= ((uint16_t)TOSH_READ_TC77_SIO_PIN() << (15 - i));
	    TOSH_uwait(5);
	    TOSH_CLR_TC77_SCK_PIN();
	}

	// cycle the clock 16 times to shift out 0xffff (power-down)
	TOSH_MAKE_TC77_SIO_OUTPUT();	
	TOSH_SET_TC77_SIO_PIN();
	for (i = 0; i < 16; i++) {
	    TOSH_uwait(5);
	    TOSH_SET_TC77_SCK_PIN();
	    TOSH_uwait(5);
	    TOSH_CLR_TC77_SCK_PIN();
	}

	// raise the CS pin to deactivate TC77
	TOSH_SET_TC77_CS_PIN();

	// we only want these bits: 0111 1111 1000 0!**
	// where 0,1 are temp, ! is status, and * is hi-Z
	data >>= 7;
	data &= 0xff;
	
	signal ADC.dataReady(data);

	flag = 0;
    }
    
    command result_t StdControl.init() {
	post init();
	return SUCCESS;
    }
    
    command result_t StdControl.start() {
	return SUCCESS;
    }
    
    command result_t StdControl.stop() {
	return SUCCESS;
    }
    
    async command result_t ADC.getData() {	
	if (flag)
	    return FAIL;

	if (flag = post start())
	    return SUCCESS;
	
	return FAIL;
    }
   
    async command result_t ADC.getContinuousData() {
	return FAIL;
    }
    
    event result_t Timer.fired() {
	post finish();
	return SUCCESS;
    }
}

  
