use strict;
use IO::Capture::Stdout;
use FindInclude;
use SlurpFile;
use Carp;

### ---
### --- PerlNesc
### ---

package PerlNesc;


sub new ($) {
  my $class = ref($_[0]) || $_[0];
  bless {
    pnflist => [],
    pnfhash => {},
    tags => { perl => \&PerlNescTag::perl_tag },
  }, $class;
}

sub add_tag {
  my ($self,$tagname,$tagsub) = @_;
  $self->{tags}->{lc($tagname)} = $tagsub;
}

sub include ($$) {
  my ($self,$pnf,$when) = @_;
  $when = 0 unless defined $when;

  if( !ref($pnf) ) {
    if( exists $self->{pnfhash}->{$pnf} ) {
      $pnf = $self->{pnfhash}->{$pnf};
    } else {
      my $name = $pnf;
      $pnf = PerlNescFile->new( name => $name, when => "" );
      push( @{$self->{pnflist}}, $self->{pnfhash}->{$name} = $pnf );
    }
  }

  if( $when ne $pnf->{when} ) {
    print STDERR "-=-=-=-=- processing $when  $pnf->{name}\n";
    $pnf->{when} = $when;
    $self->process( $pnf );
  }
}

sub reprocess ($$) {
  my ($self,$when) = @_;
  $self->include( $self->{pnflist}->[0], $when );
}

sub process ($$) {
  my ($self,$pnf) = @_;
  return unless defined $pnf->{text};
  my ($parts,$includes) = $self->process_parts($pnf,parse_text($pnf->{text}));
  $pnf->{includes} = $includes;
  return $pnf->{text} = merge_text($parts);
}

sub process_parts {
  my ($self,$pnf,$parts) = @_;
  my @includes = ();
  my $next_string_is_inc = 0;
  for my $part (@{$parts}) {

    if( exists $self->{tags}->{$part->{type}} ) {
      my $tag = $part->{type};
      if( $part->{text} =~ m{^<$tag([^>]*)>(.*)</$tag>$}s ) {
	my ($arg,$block) = ($1,$2);

	my %opts = map { m/(.*?)=(.*)/ ? ($1 => $2) : ($_ => 1) }
		   grep { /\S/ }
		   split /\s+/, $arg;
	
	if( $pnf->{when} eq "last" || !$opts{"last"} ) {

	  my $cap = new IO::Capture::Stdout;
	  $cap->start;
	  my $rv = undef;
	  eval {
	    $rv = $self->{tags}->{$tag}->(
	      pn=>$self, pnf=>$pnf, part=>$part, opts=>\%opts, text=>$block );
	  };
	  PerlNescTag::check_eval_error();
	  $cap->stop;

	  if( $rv ) {
	    my $text = join("",$cap->read);
	    my ($bparts,$bincs) = $self->process_parts( $pnf, parse_text($text) );

	    $part->{type} .= "_OUT";
	    $part->{text} = merge_text( $bparts );
	    push( @includes, @{$bincs} );

	    $pnf->{modified} = 1;
	  }

	}
      }
    } else {
      $part->{type} = "CODE";
    }

    if( $next_string_is_inc ) {
      $next_string_is_inc = 0;
      if( $part->{type} eq "STRING" && $part->{text} =~ m/^"(.*)"$/ ) {
	push( @includes, $1 );
	$self->include($1,$pnf->{when});
      }
    }

    if( $part->{type} eq "CODE" ) {
      while ($part->{text} =~ m/ \bincludes\s+(\S+)\s*;
                               | \#include\s+[<](.*?)[>]
			       | \#include\s+$
			       /gx) {
	if( defined $1 ) { push( @includes, "$1.h" ); $self->include($1,$pnf->{when}); }
	elsif( defined $2 ) { push( @includes, $2 ); $self->include($2,$pnf->{when}); }
	else { $next_string_is_inc = 1; }
      }
    } 

  }

  return ($parts,\@includes);
}

sub merge_text ($) {
  my ($parts) = @_;
  return join "", map { $_->{text} } @{$parts};
}

sub parse_text ($) {
  my ($text) = (@_);

  my $re = qr{
     ( // [^\n]* )                # $1 C++ comment
    |( /\* .*? \*/ )              # $2 C comment
    |( " (?: \\. | [^"] )* " )    # $3 quoted string
    |( <(\w+)[^>]*> .*? </\5> )   # $4 special block, $5 special keyword
    |( [^/"<]+ | . )              # $6 everything else
  }xs;

  my $code = undef;
  my @parts = ();
  while( $text =~ m{$re}g ) {
    my ($cpprem,$crem,$str,$block,$keyword,$other) = ($1,$2,$3,$4,$5,$6);

    push( @parts, { type => "COMMENT", text => $cpprem } ) if defined $cpprem;
    push( @parts, { type => "COMMENT", text => $crem } ) if defined $crem;
    push( @parts, { type => "STRING", text => $str } ) if defined $str;
    push( @parts, { type => lc $keyword, text => $block } ) if defined $block;

    if( defined $other ) {
      if( defined $code ) {
	${$code} .= $other;
      } else {
	push( @parts, { type => "CODE", text => $other } );
	$code = \$parts[-1]->{text};
      }
    } else {
      $code = undef;
    }
  }

  linenumber_parts( \@parts );

  return \@parts;
}

sub linenumber_parts {
  my ($parts) = @_;
  return unless @{$parts};
  $parts->[0]->{lines} = ($parts->[0]->{text} =~ tr/\n//);
  $parts->[0]->{linenum} = 1;
  for( my $i=1; $i<@{$parts}; $i++ ) {
    $parts->[$i]->{lines} = ($parts->[$i]->{text} =~ tr/\n//);
    $parts->[$i]->{linenum} = $parts->[$i-1]->{linenum} + $parts->[$i-1]->{lines};
  }
}


### ---
### --- PerlNescTag
### ---

package PerlNescTag;

my %Evals = ();

sub save_eval ($$) {
  eval(",");
  $Evals{$1+1} = { text => $_[0], arg => $_[1] } if $@ =~ /\(eval (\d+)\)/;
  $_[0];
}

sub check_eval_error () {
  if( $@ ) {

    my $eval = $Evals{$1} if $@ =~ /\(eval (\d+)\)/;
    die $@ if not defined $eval;

    my $n = 1;
    my $text = "     1  $eval->{text}";
    $text =~ s/\n/sprintf("\n%6d  ",++$n)/ge;

    (my $err = $@) =~ s/\s+$//;
    $err =~ s/PerlNescTag:://g;
    $err =~ s/(\(eval (\d+)\) line (\d+))/
              "$1, $Evals{$2}->{arg}{pnf}{file} line "
              . ($Evals{$2}->{arg}{part}{linenum} + $3 - 1)
	     /e;

    (my $type = $eval->{arg}{part}{type}) =~ s/_OUT$//;

    die "Error in $type tag "
      . "at $eval->{arg}{pnf}{file}:$eval->{arg}{part}{linenum}:\n"
      . "$text\n$err\n";
  }
}


sub perl_tag {
  my %arg = @_;
  my ($pn,$pnf,$part,$opts,$text) = @arg{qw(pn pnf part opts text)};
  local *include = sub { $pn->include( $_[0], $pnf->{when} ); };
  eval save_eval $text, \%arg;
  die $@ if $@;
  1;
}



### ---
### --- PerlNescFile
### ---

package PerlNescFile;

sub new ($) {
  my $class = ref($_[0]) || $_[0];
  shift;
  my %opt = @_;

  if( !defined $opt{name} && defined $opt{file} ) {
    ($opt{name} = $opt{file}) =~ s{^.*/}{};
  }

  if( !defined $opt{file} && defined $opt{name} ) {
    $opt{file} = FindInclude::find_file( $opt{name} );
  }

  if( !defined $opt{text} && defined $opt{file} ) {
    $opt{text} = SlurpFile::slurp_file( $opt{file} );
  }

  bless {
    text => $opt{text},
    name => $opt{name},
    file => $opt{file},
    when => $opt{when} || "",
    modified => 0,
    includes => [],
  }, $class;
}

1;


