#!/usr/bin/perl -w

# "Copyright (c) 2000-2002 The Regents of the University of California.  
# All rights reserved.
# 
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose, without fee, and without written agreement is
# hereby granted, provided that the above copyright notice, the following
# two paragraphs and the author appear in all copies of this software.
# 
# IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
# DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
# OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
# CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
# 
# THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
# AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS."

# Authors: Cory Sharp
# $Id$

use strict;

use FindBin;
use lib $FindBin::Bin;
use FindInclude;
use SlurpFile;



my %opts = ( ncstdin => 0 );
my %sources = ();

my $G_warning =<< 'EOF';
// *** WARNING ****** WARNING ****** WARNING ****** WARNING ****** WARNING ***
// ***                                                                     ***
// *** This file was automatically generated by create_Neighborhoods.pl.   ***
// *** Any and all changes made to this file WILL BE LOST!                 ***
// ***                                                                     ***
// *** WARNING ****** WARNING ****** WARNING ****** WARNING ****** WARNING ***

EOF

@ARGV = &FindInclude::parse_include_opts( @ARGV );
while( @ARGV && $ARGV[0] =~ /^-/ ) {
  my $opt = shift;
  if( $opt eq "-ncstdin" ) { $opts{ncstdin} = 1; }
  else { die "unknown option $opt\n"; }
}

if( $opts{ncstdin} ) {
  @ARGV = ();
  my $nn = 1;
  map { chomp; $sources{$_} = $nn++ unless defined $sources{$_} } <>;
} else {
  my $nn = 1;
  map { $sources{$_} = $nn++ unless defined $sources{$_} } @ARGV;
}

# make build/
my $dir = "build";
mkdir $dir unless -d $dir;


# process those filenames for the special lines
@ARGV = sort { $sources{$a} <=> $sources{$b} } keys %sources;
my %TypeFunc = (
  Attribute    => \&CreateAttribute,
  Neighborhood => \&CreateNeighborhood,
  Reflection   => \&CreateReflection,
  Command      => \&CreateCommand,
  NeighborhoodTypedComm => \&CreateNeighborhoodTypedComm,
);
my $TypeFuncTypes = join( "|", sort { length($b) <=> length($a) || $a cmp $b } keys %TypeFunc );
my %Interfaces = ();
while(<>) {
  my $ww = '\s*(\w+)\s*';
  if( /^\s* \/\/!! $ww = 
       \s* Create(\w+) \s*
       (?: \[ $ww (?::(.*))? \] \s*)?
       \( ([^)]*) \) \s*
       ; \s*$/ox )
  {
    my ($lval,$interface,$base,$varargs,$args) = ($1,$2,$3,$4||"",$5);
    my %uservars = ( $varargs =~ /(?:^|,)$ww=\s*(.*?)(?=,\s*\w+\s*=|$)/g );
    for my $vv (values %uservars) { $vv = trim($vv); }
    my %vars = (
	%uservars,
	Name => $lval,
	Interface => $interface, 
	Base => $base || $interface,
	Args => $args,
      );
    $TypeFunc{$interface}->( $lval, $interface, $base, $args, \%vars );
  }
}

for my $ii (values %Interfaces)
{
  if( $ii->{Interface} eq "Neighborhood" ) {
    for my $field (qw(StructFields StructInit ReflectionComponentList ReflectionStdControlWiring CommandComponentList CommandStdControlWiring)) {
      $ii->{$field} = "" unless defined $ii->{$field};
    }
  }
}

for my $ii (values %Interfaces)
{
  if( $ii->{Interface} eq "Reflection" ) {

    my $Attribute = $Interfaces{$ii->{Attribute}};
    my $Neighborhood = $Interfaces{$ii->{Neighborhood}};
    $ii->{Type} = $Attribute->{Type};
    my $field = "data_$ii->{Name}";
    my $init = $Attribute->{Init};

    $Neighborhood->{StructFields} .= "  $ii->{Type} $field;\n";
    $Neighborhood->{StructInit} .= "  $field : $init,\n" if $init ne "";
    $Neighborhood->{ReflectionComponentList} .= "$ii->{Name}C, ";
    $Neighborhood->{ReflectionStdControlWiring} .=
      "  $ii->{Neighborhood}M.ReflectionStdControl -> $ii->{Name}C.StdControl;\n";

  } elsif( $ii->{Interface} eq "Command" ) {

    my $Neighborhood = $Interfaces{$ii->{Neighborhood}};
    $Neighborhood->{CommandComponentList} .= "$ii->{Name}C, ";
    $Neighborhood->{CommandStdControlWiring} .=
      "  $ii->{Neighborhood}M.CommandStdControl -> $ii->{Name}C.StdControl;\n";

  }
}

for my $ii (values %Interfaces)
{
  sub _filter {
    my $ii = shift;
    my ($name,$base,$intr) = @$ii{qw(Name Base Interface)};
    for my $ext (@_) {
      (my $extout = $ext) =~ s/\.perl//;
      filter_nesc( find_include( "$base$ext", "$intr$ext" ), "build/$name$extout", $ii );
    }
  }

  my $jj = $ii->{Interface};
  if( $jj eq "Neighborhood" ) {
    
    _filter( $ii,
             ".perl.h", 
	     "_private.perl.nc",
	     "M.perl.nc",
	     "C.perl.nc",
	   );

    filter_nesc( find_include( "$ii->{CommBackend}M.perl.nc" ),
                 "build/$ii->{Name}CommBackendM.nc", $ii );
    filter_nesc( find_include( "$ii->{CommBackend}C.perl.nc" ),
                 "build/$ii->{Name}CommBackendC.nc", $ii );
    
  } elsif( $jj eq "Attribute" ) {
    
    _filter( $ii,
             ".perl.nc",
	     "M.perl.nc",
	     "C.perl.nc",
	     "Reflection.perl.nc",
	     "ReflectionSnoop.perl.nc",
	   );

  } elsif( $jj eq "Reflection" ) {
    
    _filter( $ii,
             "M.perl.nc",
	     "C.perl.nc",
	   );
    
  } elsif( $jj eq "Command" ) {
    
    _filter( $ii, 
             ".perl.h",
             ".perl.nc",
	     "M.perl.nc",
	     "C.perl.nc",
	   );
    
  }
}

# CreateAttribute
sub CreateAttribute {
  my ($lval,$interface,$base,$args,$vars) = @_;
  if( $args =~ /^\s* ([^=)]+) (?:=\s*(.*))? \s*$/x ) {
    my ($type,$init) = ($1,$2);
    $Interfaces{$lval} = {
      %$vars,
      Attribute => trim($lval),
      Type => trim($type),
      Init => trim($init||""),
      TypedComm => trim($lval) . "Comm",
    };
  } else {
    warn "Malformed CreateAttribute";
  }
}

sub CreateNeighborhood {
  my ($lval,$interface,$base,$args,$vars) = @_;
  my $ww = '\s*(\w+)\s*';
  if( $args =~ /^$ww,$ww,$ww,$ww$/x ) {
    my ($members,$manager,$backend,$proto) = ($1,$2,$3,$4);
    $Interfaces{$lval} = {
      %$vars,
      Neighborhood => trim($lval),
      MaxMembers => trim($members),
      Manager => trim($manager),
      CommBackend => trim($backend),
      CommProtocol => trim($proto),
      StructFields => "",
      StructInit => "",
      ReturnComponentList => "",
      ReturnStdControlWiring => "",
      CommandComponentList => "",
      CommandStdControlWiring => "",
    };
  } else {
    warn "Malformed CreateNeighborhood> $args\n";
  }
}

sub CreateReflection {
  my ($lval,$interface,$base,$args,$vars) = @_;
  my $ww = '\s*(\w+)\s*';
  if( $args =~ /^$ww,$ww,$ww,$ww,$ww$/x ) {
    my ($hood,$attr,$autopush,$dataproto,$pullproto) = ($1,$2,$3,$4,$5);
    $Interfaces{$lval} = {
      %$vars,
      Reflection => trim($lval),
      Neighborhood => trim($hood),
      Attribute => trim($attr),
      AutoPush => trim($autopush),
      DataProtocol => $dataproto,
      PullProtocol => $pullproto,
    };
  } else {
    warn "Malformed CreateReflection> $args\n";
  }
}

sub CreateCommand {
  my ($lval,$interface,$base,$args,$vars) = @_;
  my $ww = '\s*(\w+)\s*';
  my $tt = '\s*([^,]*[^,\s])\s*';
  if( $args =~ /^$ww,$tt,$tt,$ww,$ww$/x ) {
    my ($hood,$args_type,$rets_type,$args_proto,$rets_proto) = ($1,$2,$3,$4,$5);
    $Interfaces{$lval} = {
      %$vars,
      Command => trim($lval),
      Neighborhood => trim($hood),
      ArgsType => trim($args_type),
      CallProtocol => $args_proto,
      ReturnType => trim($rets_type),
      ReturnProtocol => $rets_proto,
    };
  } else {
    warn "Malformed CreateCommand";
  }
}

# filter_nesc
sub filter_nesc {
  my ($file,$outfile,$subs) = @_;
  die "ERROR, could not find $file, aborting.\n" unless defined $file;
  my $text = &SlurpFile::scrub_c_comments( &SlurpFile::slurp_file( $file ) );
  $text =~ s/^\s+//;
  my $err = "";
  $text =~ s/\$\{(\w+)\}/
	     if( defined $subs->{$1} ) { $subs->{$1}; }
	     else { $err.="$file: unknown Neighborhood substitution $1\n"; ""; }
	    /eg;
  die "${err}ERROR, unknown variables, aborting.\n" if $err;
  print STDERR "    creating $outfile from $file\n";
  &SlurpFile::dump_file( $outfile, "$G_warning$text" );
  1;
}

sub find_include {
  my $file = undef;
  for my $in (@_) {
    $file = &FindInclude::find_file( $in );
    return $file if defined $file;
  }
  die "ERROR, could not find $_[0], aborting.\n";
}

sub trim {
  my $ss = shift;
  $ss =~ s/^\s+//;
  $ss =~ s/\s+$//;
  return $ss;
}

