/**
 * Sensing a value via ADC and transmitting it over the radio.
 **/
module SenseToRfmIFXM {
   provides {
      interface StdControl;
}
   uses {
      interface MSP430ADC12External as ADC;
      interface TimerJiffy as TimeoutTimer;
      interface Leds;

      interface RadioControl;
      interface ReceiveMsg;
      interface BareSendMsg;
      interface RefVolt;
  }
}

implementation {
   norace TOS_Msg sendMsg;
   norace uint16_t counter;    

   /**
   * Initializing the components. 
   **/
   command result_t StdControl.init() {
      call Leds.init();
      return SUCCESS;
   }

   /**
   * Start the component. Send first message.
   * 
   * @return returns <code>SUCCESS</code> or <code>FAILED</code>
   **/
   command result_t StdControl.start() {
      call RefVolt.acquire(REFERENCE_2_5V);  // continue at RefVolt.stateChanged
   }
   
   event void RefVolt.stateChanged(RefVolt_t vref){
      MSP430ADC12StandardSettings_t settings = {
             externalChannel: 2,
             referenceVoltage: REFERENCE_VREFplus_AVss, 
             sampleHoldTime: SAMPLE_HOLD_64_CYCLES
      };
      call ADC.bind(settings);
      sendMsg.length = 1;
      sendMsg.data[0] = 0xFF;
      call BareSendMsg.send(&sendMsg);   // first message is dummy
   }

   command result_t StdControl.stop() {
      return SUCCESS;
   }

   /* Message sent. Now trigger new adc measurement. */
   event result_t BareSendMsg.sendDone(TOS_MsgPtr msg, result_t success){
      //call Leds.yellowToggle();
      //return call ADC.getSingleData();
      return call TimeoutTimer.setOneShot(500);
   }  

   /**
   * ADC measurement done.
   * Now send the adc-data out over the radio. We will get the 
   * BareSendMsg.sendDone event after transmission.
   **/
   event void ADC.dataReadySingle(uint16_t data) {
      uint8_t i;
      sendMsg.length = TOSH_DATA_LENGTH;
      // send 8 msbits
      sendMsg.data[0] = 0x00FF & data;
      sendMsg.data[1] = data>>8;
      for(i=2; i<sendMsg.length; i++)
         sendMsg.data[i] = 0xF0;
      call BareSendMsg.send(&sendMsg);
   }
   
   /* We will not receive any messages */
   event TOS_MsgPtr ReceiveMsg.receive(TOS_MsgPtr m){
      return m;
   }
   
   event result_t TimeoutTimer.fired() {
      call Leds.yellowToggle();
      return call ADC.getSingleData();
   }
   
   async event void ADC.dataReadySingleRpt(uint16_t data) {}
   event void ADC.dataReadySequence() {}
   async event void ADC.dataReadySequenceRpt() {}
}
