/*
 * Copyright (c) 2004, Technische Universitaet Berlin
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - Neither the name of the Technische Universitaet Berlin nor the names
 *   of its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * - Description ---------------------------------------------------------
 * Controlling the TDA5250, switching modes and initializing.
 * - Revision -------------------------------------------------------------
 * $Revision$
 * $Date$
 * @author: Jan Hauer (hauer@tkn.tu-berlin.de)
 * ========================================================================
 */

module TDA5250ControlM {
  provides {
    interface TDA5250Control;
    interface StdControl;
  }
  uses interface HPLUSARTControl as USART;
  uses interface TimerJiffy as DelayTimer;
}

implementation {

   // default: switching radio state (TX,RX,Sleep and ASK,FSK) is done externally (Pin).
   // If this should be done via controlword (SPI), MODESWITCH_SPI_CONTROLLED must
   // be defined.
   //#define MODESWITCH_SPI_CONTROLLED
   
   uint16_t currentConfig;
   TDA5250Delays_t currentDelay;
   
   /* Set the exact contents of the radio data registers  */
   void setCONFIG(uint16_t value);
   void setFSK(uint16_t value);
   void setXTAL_TUNING(uint16_t value);
   void setLPF(uint8_t value);
   void setON_TIME(uint16_t value);
   void setOFF_TIME(uint16_t value);
   void setCOUNT_TH1(uint16_t value);
   void setCOUNT_TH2(uint16_t value);
   void setRSSI_TH3(uint8_t value);
   void setCLK_DIV(uint8_t value);
   void setXTAL_CONFIG(uint8_t value);
   void setBLOCK_PD(uint16_t value);
   
   /* Read the exact contents of the radio data registers */
   uint8_t getSTATUS();
   uint8_t getADC();
   
   /* Reading and writing to the radio over the USART */
   void transmitByte(uint8_t data);
   void writeByte(uint8_t address, uint8_t data);
   void writeWord(uint8_t address, uint16_t data);
   uint8_t readByte(uint8_t address);
   
   /* Other internal functions */
   result_t SetDelayTimer(TDA5250Delays_t type);
   
   /* Modes of operation for the radio in Slave Mode*/
   enum {
      RADIO_MODE_TX,
      RADIO_MODE_RX,
      RADIO_MODE_SLEEP
   } radioMode;
   
   /************** exported commandhandlers ***************/
   TOSH_SIGNAL(PORT1_VECTOR) {
      P1IFG &= 0xFFFE;   // Reset interrupt flag for the PWDD Pin
       signal TDA5250Control.ValidDataDetected();
   }
   
   /* all values to default */
   command void TDA5250Control.reset(){
      call USART.setModeSPI();

      currentConfig = DATA_CONFIG_DEFAULT;
      #ifdef MODESWITCH_SPI_CONTROLLED
         CONFIG_CONTROL_TXRX_REGISTER(currentConfig);
      #else
         CONFIG_CONTROL_TXRX_EXTERNAL(currentConfig);
      #endif
      writeWord(ADRW_CONFIG, currentConfig);
      writeWord(ADRW_FSK, DATA_FSK_DEFAULT);
      writeWord(ADRW_XTAL_TUNING, DATA_XTAL_TUNING_DEFAULT);
      writeByte(ADRW_LPF, DATA_LPF_DEFAULT);
      writeWord(ADRW_ON_TIME, DATA_ON_TIME_DEFAULT);
      writeWord(ADRW_OFF_TIME, DATA_OFF_TIME_DEFAULT);
      writeWord(ADRW_COUNT_TH1, DATA_COUNT_TH1_DEFAULT);
      writeWord(ADRW_COUNT_TH2, DATA_COUNT_TH2_DEFAULT);
      writeByte(ADRW_RSSI_TH3, DATA_RSSI_TH3_DEFAULT);
      writeByte(ADRW_CLK_DIV, DATA_CLK_DIV_DEFAULT);
      writeByte(ADRW_XTAL_CONFIG, DATA_XTAL_CONFIG_DEFAULT);
      writeWord(ADRW_BLOCK_PD, DATA_BLOCK_PD_DEFAULT);
      atomic radioMode = RADIO_MODE_RX;
      
      TOSH_uwait(12000); // wait 12ms for the initial startup
      //SetDelayTimer(SYSTEM_SETUP);
   }

   command result_t StdControl.init(){         
      // enabling pins for module / io function
      TOSH_SEL_TDA_BUSM_IOFUNC();
      TOSH_SEL_TDA_ENTDA_IOFUNC();
      TOSH_SEL_TDA_TXRX_IOFUNC();
      TOSH_SEL_TDA_DATA_IOFUNC();
      TOSH_SEL_TDA_PWDDD_IOFUNC();

      // setting direction
      TOSH_MAKE_TDA_BUSM_OUTPUT();
      TOSH_MAKE_TDA_ENTDA_OUTPUT();
      TOSH_MAKE_TDA_TXRX_OUTPUT();

      TOSH_MAKE_TDA_DATA_OUTPUT();  // no interference from timerA-pin
      TOSH_CLR_TDA_DATA_PIN();
      TOSH_MAKE_TDA_DATA_INPUT();

      // Default as output pin, but changes to input
      // with SELF_POLLING or TIMER modes
      TOSH_MAKE_TDA_PWDDD_OUTPUT();

      // initializing the radio
      TOSH_SET_TDA_BUSM_PIN();  // set busmode: SPI for TDA
      TOSH_SET_TDA_ENTDA_PIN();
      TOSH_SET_TDA_TXRX_PIN();
      TOSH_CLR_TDA_PWDDD_PIN();
      
      call TDA5250Control.reset();

      return SUCCESS;
  }

   command result_t StdControl.start(){
      return SUCCESS;
   }

   command result_t StdControl.stop(){
      return SUCCESS;
   }
   
   command void TDA5250Control.RxMode(){
      if (radioMode != RADIO_MODE_RX) {
         #ifdef MODESWITCH_SPI_CONTROLLED
            call USART.setModeSPI();
            CONFIG_RX_NTX_RX(currentConfig);
            writeWord(ADRW_CONFIG, currentConfig);
         #else
            TOSH_CLR_TDA_PWDDD_PIN();
            TOSH_SET_TDA_TXRX_PIN();
         #endif
         
         switch(radioMode)
         {
            // TODO!
            case RADIO_MODE_RX: 
               break;
            case RADIO_MODE_TX:
            case RADIO_MODE_SLEEP: 
               //TOSH_uwait(2860); //wait 2.86ms for TX->RX (PD->RX)
               SetDelayTimer(RECEIVER_SETUP);
               break;
         }
         radioMode = RADIO_MODE_RX;
      }
      else signal TDA5250Control.ReceiverSetupDone();
   }

   command void TDA5250Control.TxMode(){
      if (radioMode != RADIO_MODE_TX) {
         #ifdef MODESWITCH_SPI_CONTROLLED
            call USART.setModeSPI();
            CONFIG_RX_NTX_TX(currentConfig);
            writeWord(ADRW_CONFIG, currentConfig);
         #else
            TOSH_CLR_TDA_PWDDD_PIN();
            TOSH_CLR_TDA_TXRX_PIN();
         #endif

         switch(radioMode)
         {
            // TODO!
            case RADIO_MODE_TX: 
               break;
            case RADIO_MODE_RX:
            case RADIO_MODE_SLEEP: 
               //TOSH_uwait(1430); //wait 1.43ms for RX->TX (PD->TX)
               SetDelayTimer(TRANSMITTER_SETUP);
               break;
         }
      }
      else signal TDA5250Control.TransmitterSetupDone();
   }

   command void TDA5250Control.SleepMode(){
      if (radioMode != RADIO_MODE_SLEEP) {
         #ifdef MODESWITCH_SPI_CONTROLLED
            call USART.setModeSPI();
            CONFIG_ALL_PD_POWER_DOWN(currentConfig);
            writeWord(ADRW_CONFIG, currentConfig);
         #else
            TOSH_SET_TDA_PWDDD_PIN();
         #endif
         radioMode = RADIO_MODE_SLEEP;
      }
   }
   
   command result_t TDA5250Control.SetRSSIThreshold(uint8_t value) {
      setRSSI_TH3(0xC0 | value);
      return SUCCESS;
   }
   
   command uint8_t TDA5250Control.GetRSSIValue () {
      return (0x3F & getADC());
   }
   
   command result_t TDA5250Control.SetOutputFrequency(TDA5250ClockOutFreq_t freq) {
      setCLK_DIV(freq);
   }
   
   /**
      Set the on time and off time of the radio
      (Only makes sense when in TIMER or SELF_POLLING Mode)
   */
   command result_t TDA5250Control.SetOnTime_ms(float time) {
      setON_TIME(CONVERT_TIME(time));
      return SUCCESS;
   }
   command result_t TDA5250Control.SetOffTime_ms(float time) {
      setOFF_TIME(CONVERT_TIME(time));
      return SUCCESS;
   }
   
   /**
      Set the mode of the radio 
      The choices are SLAVE_MODE, TIMER_MODE, SELF_POLLING_MODE
   */
   command result_t TDA5250Control.SetMode(TDA5250Mode_t mode) {
      switch (mode) {
         case SLAVE_MODE:
            call TDA5250Control.SetSlaveMode();
            return SUCCESS;
         case SELF_POLLING_MODE:
            call TDA5250Control.SetSelfPollingMode(100,100);
            return SUCCESS;
         case TIMER_MODE:
            call TDA5250Control.SetTimerMode(100,100);
            return SUCCESS;
         default:
            return SUCCESS;
      }
      return SUCCESS;
   }
   
   command result_t TDA5250Control.SetSlaveMode() {
      P1IE &= 0xFFFE; // Disable Interrupts on the PWDD Pin
      CONFIG_MODE_1_SLAVE_OR_TIMER(currentConfig);
      CONFIG_MODE_2_SLAVE(currentConfig);
      CONFIG_D_OUT_ALWAYS(currentConfig);
      setCONFIG(currentConfig);
      TOSH_MAKE_TDA_PWDDD_OUTPUT();
      return SUCCESS;
   }
   
   command result_t TDA5250Control.SetTimerMode(float on_time, float off_time) {
      P1IE |= 0x1; // Enable Interrupts on the PWDD Pin
      CONFIG_MODE_1_SLAVE_OR_TIMER(currentConfig);
      CONFIG_MODE_2_TIMER(currentConfig);
      CONFIG_D_OUT_IFVALID(currentConfig);
      setCONFIG(currentConfig);
      setON_TIME(CONVERT_TIME(on_time));
      setOFF_TIME(CONVERT_TIME(off_time));
      TOSH_MAKE_TDA_PWDDD_INPUT();
      return SUCCESS;
   }
   
   command result_t TDA5250Control.SetSelfPollingMode(float on_time, float off_time) {
      P1IE |= 0x1; // Enable Interrupts on the PWDD Pin
      CONFIG_MODE_1_SELF_POLLING(currentConfig);
      CONFIG_D_OUT_IFVALID(currentConfig);
      setCONFIG(currentConfig);
      setON_TIME(CONVERT_TIME(on_time));
      setOFF_TIME(CONVERT_TIME(off_time));
      TOSH_MAKE_TDA_PWDDD_INPUT();
      return SUCCESS;
   }
   
   /*
      Timer used for any delay needed when making transitions of 
      state in the radio
   */
   event result_t DelayTimer.fired() {
      switch(currentDelay) {
         case SYSTEM_SETUP:
            signal TDA5250Control.SystemSetupDone(); break;
         case RECEIVER_SETUP:
            atomic radioMode = RADIO_MODE_RX;
            signal TDA5250Control.ReceiverSetupDone(); break;
         case DATA_DETECTION_SETUP:
            signal TDA5250Control.DataDetectionSetupDone(); break;
         case RSSI_STABLE:
            signal TDA5250Control.RSSIStableDone(); break;
         case DATA_VALID:
            signal TDA5250Control.DataValidDone(); break;
         case CLOCK_OUT_SETUP:
            signal TDA5250Control.ClockOutSetupDone(); break;
         case TRANSMITTER_SETUP:
            atomic radioMode = RADIO_MODE_TX;
            signal TDA5250Control.TransmitterSetupDone(); break;
         case XTAL_STARTUP:
            signal TDA5250Control.XTALStartupDone(); break;
         default:
            return FAIL;
      }
      return SUCCESS;
   }
   
   /*
   ############################################################
   ##################### Internal Functions ###################
   ############################################################
   */
   
   /* Reading and writing to the radio over the USART */
   void transmitByte(uint8_t data){
      call USART.tx(data);
      while (call USART.isTxIntrPending() == FAIL);
   }

   void writeByte(uint8_t address, uint8_t data) {
      TOSH_CLR_TDA_ENTDA_PIN();
      transmitByte(address);
      transmitByte(data);
      while (call USART.isRxIntrPending() == FAIL);
      TOSH_SET_TDA_ENTDA_PIN();
      return;
   }

   void writeWord(uint8_t address, uint16_t data){
      TOSH_CLR_TDA_ENTDA_PIN();
      transmitByte(address);
      transmitByte((uint8_t) (data >> 8));
      transmitByte((uint8_t) data);
       while (call USART.isRxIntrPending() == FAIL);
      TOSH_SET_TDA_ENTDA_PIN();
      return;
   }

   uint8_t readByte(uint8_t address){
      TOSH_CLR_TDA_ENTDA_PIN();
      transmitByte(address);
      while (call USART.isRxIntrPending() == FAIL);
      TOSH_SET_TDA_ENTDA_PIN();
      return call USART.rx();
   }
   
   /* Read the exact contents of the radio data registers */
   uint8_t getSTATUS() {
      call USART.setModeSPI();
      return readByte(ADRR_STATUS);      
   }
   uint8_t getADC() {
      call USART.setModeSPI();
      return readByte(ADRR_ADC);   
   }
    
   /* Set the exact contents of the radio data registers  */
   void setCONFIG(uint16_t value) {
      call USART.setModeSPI();
      writeWord(ADRW_CONFIG, value);
   }
   void setFSK(uint16_t value) {
      call USART.setModeSPI();
      writeWord(ADRW_FSK, value);
   }
   void setXTAL_TUNING(uint16_t value) {
      call USART.setModeSPI();
      writeWord(ADRW_XTAL_TUNING, value);
   }
   void setLPF(uint8_t value) {
      call USART.setModeSPI();
      writeByte(ADRW_LPF, value);
   }
   void setON_TIME(uint16_t value) {
      call USART.setModeSPI();
      writeWord(ADRW_ON_TIME, value);
   }
   void setOFF_TIME(uint16_t value) {
      call USART.setModeSPI();
      writeWord(ADRW_FSK, value);
   }
   void setCOUNT_TH1(uint16_t value) {
      call USART.setModeSPI();
      writeWord(ADRW_COUNT_TH1, value);
   }
   void setCOUNT_TH2(uint16_t value) {
      call USART.setModeSPI();
      writeWord(ADRW_COUNT_TH2, value);
   }
   void setRSSI_TH3(uint8_t value) {
      call USART.setModeSPI();
      writeByte(ADRW_RSSI_TH3, value);
   }
   void setCLK_DIV(uint8_t value) {
      call USART.setModeSPI();
      writeByte(ADRW_CLK_DIV, value);
   }
   void setXTAL_CONFIG(uint8_t value) {
      call USART.setModeSPI();
      writeByte(ADRW_XTAL_CONFIG, value);
   }
   void setBLOCK_PD(uint16_t value) {
      call USART.setModeSPI();
      writeWord(ADRW_BLOCK_PD, value);
   }
   
   /* Other internal functions */

   result_t SetDelayTimer(TDA5250Delays_t type) {
      uint16_t delay;
      bool temp;
      atomic currentDelay = type;
      switch(type) {
         case SYSTEM_SETUP:
            delay = SYSTEM_SETUP_TIME;
            break;
         case RECEIVER_SETUP:
            delay = RECEIVER_SETUP_TIME;
            break;
         case DATA_DETECTION_SETUP:
            delay = DATA_DETECTION_SETUP_TIME;
            break;
         case RSSI_STABLE:
            delay = RSSI_STABLE_TIME;
            break;
         case DATA_VALID:
            delay = DATA_VALID_TIME;
            break;
         case CLOCK_OUT_SETUP:
            delay = CLOCK_OUT_SETUP_TIME;
            break;
         case TRANSMITTER_SETUP:
            delay = TRANSMITTER_SETUP_TIME;
            break;
         case XTAL_STARTUP:
            delay = XTAL_STARTUP_TIME;
            break;
         default:
            return FAIL;
      }
      while(call DelayTimer.setOneShot(delay) == FAIL);
      return SUCCESS;
   }

}



