/*
 * Copyright (c) 2004, Technische Universitaet Berlin
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - Neither the name of the Technische Universitaet Berlin nor the names
 *   of its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * - Description ----------------------------------------------------------
 * Module for sending and receiving byte-streams over the TDA5250.
 * Currently (preliminary, for testing) this is implemented:
 * The emitted phy-frame consists of a preamble, start-frame-delimiter,
 * a length field, and the payload plus crc16:
 *                -------------------------------------------------
 * frame format: |     Preamble   | SFD | Length | Payload | CRC16 |
 *                -------------------------------------------------
 * bytes:           SYNS_TO_SEND    1       1         ?       2
 * Note: Each transmitted byte is encapsulated in a UART-Frame.
 * Currently no channel encoding is used.
 * - Revision -------------------------------------------------------------
 * $Revision$
 * $Date$
 * @author: Jan Hauer (hauer@tkn.tu-berlin.de)
 * ========================================================================
 */
includes crc;

module TDA5250RadioM {
   provides {
      interface StdControl;
      interface RadioControl;
      interface ReceiveMsg as Receive;
      interface BareSendMsg as Send;
   }
   uses {
      interface HPLUART;
      interface TDA5250Control;
      interface HPLUSARTControl as USARTControl;
      interface HPLUSARTFeedback as USARTData;
      interface LocalTime;
      interface TimerJiffy as AckTimeout;
      interface TimerJiffy as BackOffTimer;
      interface ChannelMonitor as CSMA;
      interface Random;
      command result_t setRSSIThreshold(uint16_t level);
   }
}
implementation
{
   /**************** Module Definitions  *****************/
   #define PREAMBLE_BYTE                  0xAA
   #define SYNC_BYTE                      0xFF
   #define SFD_BYTE                       0x33
   #define PREAMBLES_TO_SEND              2
   #define INIT_BACKOFF_SHIFT             6
   #define CHANNEL_BUSY_BACKOFF_SHIFT     6
   #define ACK_LENGTH                     2
   #define ACK_CODE_1                     0xBA
   #define ACK_CODE_2                     0x83
   #define ACK_TIMEOUT                    1000   // 1 Jiffy = 1/32 kHz                

   typedef enum {
      INIT_STATE,
      TX_STATE,
      RX_STATE
   } radioState_t;

   typedef enum {
      TX_STATE_NULL,
      TX_STATE_PREAMBLE,
      TX_STATE_SYNC,
      TX_STATE_SFD,
      TX_STATE_DATA,
      TX_STATE_TIME1,
      TX_STATE_TIME2,
      TX_STATE_CRC1,
      TX_STATE_CRC2,
      TX_STATE_WAIT_FOR_ACK,
      TX_STATE_READ_ACK,
      TX_STATE_DONE
   } radioTxState_t;

   typedef enum {
      RX_STATE_NULL,
      RX_STATE_PREAMBLE,
      RX_STATE_SYNC,
      RX_STATE_SFD,
      RX_STATE_DATA,
      RX_STATE_TIME1,
      RX_STATE_TIME2,
      RX_STATE_CRC1,
      RX_STATE_CRC2
   } radioRxState_t;

   /**************** Module Global Variables  *****************/
   TOS_Msg ackMsg;
   TOS_Msg rxBuf;        // receive buffer
   TOS_Msg *txBufPtr;  // pointer to transmit buffer
   TOS_Msg *rxBufPtr;  // pointer to receive buffer
   bool bTxBusy;
   radioState_t radioState;   // Current Radio State
   radioTxState_t radioTxState;   // Current Radio Tx State
   radioRxState_t radioRxState;   // Current Radio Rx State
   uint16_t txByteCnt;      // index into current tx data
   uint16_t rxByteCnt;      // index into current rx data
   uint16_t runningTxCRC;   // crc attached on sent message
   uint16_t preamblesSent;      // number of Sync bytes sent so far
   uint8_t nextTxByte;      // Next byte to be transmitted
   uint16_t txMsgLength;   // Length of message to transmit... including header info
   uint16_t rxMsgLength;   // Length of message to receive ... including header info
   uint32_t localTime; // Local time since the node was initialized or last sent a msg
   
   uint8_t sendingAck;
   uint8_t waitingForAck;
   
   bool waitForRx;

   /**************** Local Function Declarations  *****************/
   result_t InitState();
   result_t RadioInit();
   result_t SetRxMode();
   result_t SetTxMode();
   result_t SendCurrentMsg();
   result_t RecvMsg(TOS_Msg* msg);
   result_t TransmitNextByte();
   result_t ReceiveNextByte(uint8_t data);
   result_t SetRandomBackoffTimer(uint16_t rand_div);

   /**************** Tasks that are Posted  *****************/
   task void PacketSent();
   task void PacketRcvd();

   /**************** Radio Init  *****************/
   command result_t StdControl.init(){
      return InitState();
   }

   /**************** Radio Reset  *****************/
   command result_t RadioControl.SetInitState(){
      return InitState();
   }

   /**************** Radio Start  *****************/
   command result_t StdControl.start(){
      waitForRx = TRUE;
      SetRxMode();
      while(waitForRx == TRUE);
      return SUCCESS;
   }

   /**************** Radio Stop  *****************/
   command result_t StdControl.stop(){
      return SUCCESS;
   }

   /**************** Radio Control ****************/
   command result_t RadioControl.setRxMode(){
      waitForRx = TRUE;
      SetRxMode();
      while(waitForRx == TRUE);
      return SUCCESS;
   }

   async event result_t CSMA.queryChannelComplete(channel_status_t status) {
      if (status == CHANNEL_FREE)
         return SendCurrentMsg();
      else {
         TOSH_SET_LED0_PIN();
         return SetRandomBackoffTimer(CHANNEL_BUSY_BACKOFF_SHIFT);
      }
   }

   /**************** Radio Send ****************/
   command result_t Send.send(TOS_Msg *msg) {
   result_t Result = SUCCESS;
      atomic {
         if (bTxBusy) {
            Result = FAIL;
         }
         else {
            bTxBusy = TRUE;
            txBufPtr = msg;
         }
      }
      if (Result) {
         atomic radioState = TX_STATE;
         call USARTControl.disableRxIntr();
         call USARTControl.disableTxIntr();
         waitForRx = TRUE;
         call TDA5250Control.RxMode();
         while(waitForRx == TRUE);
         //Once the ReceiverSetupDone event is signalled, we query the channel for CSMA
      }
      return Result;
   }

   /**************** USART Tx Done ****************/
   async event result_t USARTData.txDone() {
      return TransmitNextByte();
   }

   
   /**************** USART Rx Done ****************/
   async event result_t USARTData.rxDone(uint8_t data) {
      //call HPLUART.put(data);
      ReceiveNextByte(data);
         //call HPLUART.put(0x77);
   }

   /********* Timeout for receiving acknowledgements *********/
   event result_t AckTimeout.fired() {
      atomic {
         waitingForAck = 0;
         txBufPtr->ack = 0;
         signal Send.sendDone(txBufPtr, SUCCESS);
      }
      return SUCCESS;
   }

   /********* Timeout and Backoff for CSMA *********/
   event result_t BackOffTimer.fired() {
      TOSH_CLR_LED0_PIN();
      return call CSMA.queryChannel();
   }

   /**
      Signals that the system is finished changing states
   */
   async event result_t TDA5250Control.SystemSetupDone() {return SUCCESS;}
   async event result_t TDA5250Control.RadioSetupDone(){return SUCCESS;}
   async event result_t TDA5250Control.ReceiverSetupDone(){
      atomic waitForRx = FALSE;
      switch(radioState) {
         case INIT_STATE:        
         case RX_STATE:
            call USARTControl.setModeUART_RX();
            return call USARTControl.enableRxIntr();
         case TX_STATE:
            //SetRandomBackoffTimer(INIT_BACKOFF_SHIFT);
            return call CSMA.queryChannel();
         default:
            return FAIL; //NEVER SHOULD BE HERE
      }
   }
   async event result_t TDA5250Control.DataDetectionSetupDone(){return SUCCESS;}
   async event result_t TDA5250Control.RSSIStableDone(){return SUCCESS;}
   async event result_t TDA5250Control.DataValidDone(){return SUCCESS;}
   async event result_t TDA5250Control.ClockOutSetupDone(){return SUCCESS;}
   async event result_t TDA5250Control.TransmitterSetupDone(){
      call USARTControl.setModeUART_TX();
      call USARTControl.enableTxIntr();
      atomic localTime = 0;//call LocalTime.read();
      TransmitNextByte();
      return SUCCESS;
   }
   async event result_t TDA5250Control.XTALStartupDone(){return SUCCESS;}
   async event uint16_t TDA5250Control.ValidDataDetected() {return SUCCESS;}
   
   event async result_t HPLUART.get(uint8_t data) {
     //call Leds.greenToggle();
     return SUCCESS;
   }

   event async result_t HPLUART.putDone() {
     return SUCCESS;
   }

   /**************** Local Function Definitions  *****************/
   result_t InitState() {
      waitForRx = FALSE;
      call HPLUART.init();
      call Random.init();
      return RadioInit();
   }
   
   /* Initialize the radio and initialize all global variables */
   result_t RadioInit() {
      call TDA5250Control.reset();
      call TDA5250Control.SetSlaveMode();
      atomic {
         ackMsg.length = ACK_LENGTH;
         ackMsg.data[0] = ACK_CODE_1;
         ackMsg.data[1] = ACK_CODE_2;
         radioState= INIT_STATE;
         radioTxState = TX_STATE_NULL;
         radioRxState = RX_STATE_NULL;
         txBufPtr = NULL;
         rxBufPtr = &rxBuf;
         txByteCnt = 0;
         rxByteCnt = 0;
         runningTxCRC = 0;
         sendingAck = 0;
         waitingForAck = 0;
         txMsgLength = 0;
         rxMsgLength = 0;
      }
      return SUCCESS;
   }

   /* Set the Radio into RX mode and set up all state information and variables
      to be ready to receive */
   result_t SetRxMode() {
      atomic {
         radioState = RX_STATE;
         radioRxState = RX_STATE_SFD;
         rxByteCnt = 0;
         rxBuf.length = 0;
         rxBuf.crc = 0;
         rxMsgLength = 0;

      }
      call USARTControl.disableRxIntr();
      call USARTControl.disableTxIntr();      
      call TDA5250Control.RxMode();
      //Wait for event signalling transmit mode of radio, and then turn interrupts back on
      return SUCCESS;
   }

   /* Set the Radio into TX mode and set up all state information and variables
      to be ready to transmit */
   result_t SetTxMode() {
      atomic {
         radioState = TX_STATE;
         radioTxState = TX_STATE_PREAMBLE;
         txByteCnt = 0;
         runningTxCRC = 0;
         txMsgLength = 0;
      }
      call USARTControl.disableTxIntr();
      call USARTControl.disableRxIntr();
      call TDA5250Control.TxMode();
      //Wait for event signalling transmit mode of radio, and then turn interrupts back on
      return SUCCESS;
   }

   /* Begins Sending a message over the USART */
   result_t SendCurrentMsg() {
      //Wait for TransmitterSetupDone event to signal, and then start transmitting bytes
      return SetTxMode();
   }

   /* Posted once a message is completely sent */
   task void PacketSent() {
      TOS_MsgPtr pBuf;
      atomic {
          txBufPtr->time = 0;
          pBuf = txBufPtr;
      }
      atomic bTxBusy = FALSE;
      signal Send.sendDone((TOS_MsgPtr)pBuf, SUCCESS);

   /*
      if (sendingAck == 1) {
         sendingAck = 0;
         rxBufPtr = signal Receive.receive(rxBufPtr);
      }
      else {
         waitingForAck = 1;
         SetRxMode();
         call AckTimeout.setOneShot(ACK_TIMEOUT);
         //signal Send.sendDone(txBufPtr, SUCCESS);
      }
   */
   }

   /* Posted once a message is completely received */
   task void PacketRcvd() {
      TOS_MsgPtr pBuf;
      atomic {
         rxBufPtr->time = 0;
         pBuf = rxBufPtr;
      }
      pBuf = signal Receive.receive((TOS_MsgPtr)pBuf);

      atomic {
         if (pBuf)
            rxBufPtr = pBuf;
         rxBufPtr->length = 0;
      }

    /*
      if (waitingForAck == 1) {
         if(rxBufPtr->data[0] == ACK_CODE_1 && rxBufPtr->data[1] == ACK_CODE_2) {
            atomic {
               waitingForAck = 0;
               txBufPtr->ack = 1;
               signal Send.sendDone(txBufPtr, SUCCESS);
            }
            return;
         }
      }
      sendingAck = 1;
      atomic txBufPtr = ackMsg;
      SendCurrentMsg();
   */
   }

   /* transmit the next Byte over the USART */
   result_t TransmitNextByte() {
      result_t ret_val = SUCCESS;
      atomic {
         switch(radioTxState) {
            case TX_STATE_PREAMBLE:
               if(preamblesSent < PREAMBLES_TO_SEND-1)
                  preamblesSent++;
               else {
                  radioTxState = TX_STATE_SYNC;
                  preamblesSent = 0;
               }
               ret_val = call USARTControl.tx(PREAMBLE_BYTE);
               break;
            case TX_STATE_SYNC:
               radioTxState = TX_STATE_SFD;
               ret_val = call USARTControl.tx(SYNC_BYTE);
               break;
            case TX_STATE_SFD:
               radioTxState = TX_STATE_DATA;
               txMsgLength = offsetof(struct TOS_Msg,data) + txBufPtr->length;
               txByteCnt = 0;
               runningTxCRC = 0;
               ret_val = call USARTControl.tx(SFD_BYTE);
               break;
            case TX_STATE_DATA:
               if (txByteCnt == txMsgLength - 1)
                  radioTxState = TX_STATE_TIME1;
               nextTxByte = ((uint8_t*)txBufPtr)[txByteCnt];
               runningTxCRC = crcByte(runningTxCRC, nextTxByte);
               txByteCnt++;
               ret_val = call USARTControl.tx(nextTxByte);
               break;
            case TX_STATE_TIME1:
               radioTxState = TX_STATE_TIME2;
               runningTxCRC = crcByte(runningTxCRC,(uint8_t)(localTime >> 8));
               ret_val = call USARTControl.tx((uint8_t)(localTime >> 8));
               break;
            case TX_STATE_TIME2:
               radioTxState = TX_STATE_CRC1;
               runningTxCRC = crcByte(runningTxCRC,(uint8_t)localTime);
               ret_val = call USARTControl.tx((uint8_t)localTime);
               break;
            case TX_STATE_CRC1:
               radioTxState = TX_STATE_CRC2;
               ret_val = call USARTControl.tx((uint8_t) (runningTxCRC >> 8));
               break;
            case TX_STATE_CRC2:
               radioTxState = TX_STATE_DONE;
               ret_val = call USARTControl.tx((uint8_t)(runningTxCRC));
               break;
            case TX_STATE_DONE:
               radioTxState = TX_STATE_NULL;
               while(call USARTControl.isTxEmpty() == FAIL);
               SetRxMode();
               post PacketSent();
               break;
            default:
               ret_val = FAIL;
               break;
         }
      }
      return ret_val;
   }

   /* Receive the next Byte from the USART */
   result_t ReceiveNextByte(uint8_t data) {
      result_t ret_val = SUCCESS;
      atomic {
         switch(radioRxState) {
            case RX_STATE_NULL:
            case RX_STATE_PREAMBLE:
               if(data == PREAMBLE_BYTE)
                  radioRxState = RX_STATE_SYNC;
               break;
            case RX_STATE_SYNC:
               if(data != PREAMBLE_BYTE)
                  radioRxState = RX_STATE_SFD;
               break;
            case RX_STATE_SFD:         
               if (data == SFD_BYTE) {
                  rxByteCnt = 0;
                  rxBufPtr->crc = 0;
                  rxMsgLength = offsetof(struct TOS_Msg,length);
                  radioRxState = RX_STATE_DATA;
               }
               else {
                  radioRxState = RX_STATE_NULL;
                  ret_val = FAIL;
               }
               break;             
            case RX_STATE_DATA:
               if(rxByteCnt == offsetof(struct TOS_Msg,length))
                  if(data <= TOSH_DATA_LENGTH)
                     rxMsgLength += data;
                  else {
                     radioRxState = RX_STATE_NULL;
                     ret_val = FAIL;
                     break;
                  }
               if(rxByteCnt == rxMsgLength)
                  radioRxState = RX_STATE_TIME1;
               ((uint8_t*)rxBufPtr)[rxByteCnt++] = data;
               rxBufPtr->crc = crcByte(rxBufPtr->crc, data);
               break;
            case RX_STATE_TIME1:
               radioRxState = RX_STATE_TIME2;
               rxBufPtr->time = ((uint16_t)data) << 8;
               rxBufPtr->crc = crcByte(rxBufPtr->crc, data);
               break;
            case RX_STATE_TIME2:
               radioRxState = RX_STATE_CRC1;
               rxBufPtr->time |= (uint16_t)data;
               rxBufPtr->crc = crcByte(rxBufPtr->crc, data);
               break;
            case RX_STATE_CRC1:
               if (data == (uint8_t) (rxBufPtr->crc >> 8)) {
                  radioRxState = RX_STATE_CRC2;
               }
               else {
                  rxBufPtr->crc = 0;
                  radioRxState = RX_STATE_NULL;
                  ret_val = post PacketRcvd();
               }
               break;
            case RX_STATE_CRC2:
               if (data == (uint8_t)(rxBufPtr->crc)){
                  rxBufPtr->crc = 1;
               }
               else {
                  rxBufPtr->crc = 0;
               }
               radioRxState = RX_STATE_NULL;
               ret_val = post PacketRcvd();
               break;
            default:
               radioRxState = RX_STATE_NULL;
               ret_val = FAIL;
               break;
         }
      }
      return ret_val;
   }
   
   result_t SetRandomBackoffTimer(uint16_t rand_shift) {
      uint16_t randomBackoff;
      randomBackoff = (call Random.rand()) >> rand_shift;
      return call BackOffTimer.setOneShot(randomBackoff);
   }
}

