/*
 * CompassControl.java
 * (c) 2001 by Phil Scovis, all rights reserved.
 * This code is provided as-is, without warranty.
 * You may copy, use, or distribute this code,
 * provided that you add value to it and include
 * this notice.
 *
 * Modified on February 1, 2003, 2:53 PM (Peter Volgyesi):
 *  - converted to Swing
 */

import java.awt.SystemColor;
import java.awt.Graphics;
import java.awt.Dimension;
import java.awt.Polygon;
import java.awt.Font;
import java.awt.FontMetrics;

/**
 *
 * A component that displays a graphical representation of a compass.  The
 * compass "swings" when the setting is changed.
 * @author <a href="mailto:phils@altavista.com">Phil Scovis</a>
 * @author <a href="mailto:peter.volgyesi@vanderbilt.edu">Peter Volgyesi</a>
 */
public class CompassControl extends javax.swing.JPanel {

    /** Creates new panel CompassControl */
    public CompassControl() {
        setAngle(N);
        initComponents();
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    private void initComponents() {//GEN-BEGIN:initComponents

        setLayout(new java.awt.BorderLayout());

    }//GEN-END:initComponents

    /**
     * Sets the angle of the compass.  This causes the compass to "swing" from
     * its current position to the new position.  The value set here is
     * immediately reflected in a subsequent call to <code>getAngle()</code> even
     * if the compass is still swinging.
     * @param angle the new angle of the compass, in radians.  Values are clockwise
     * beginning from North.
     * @see #getAngle()
     */
    public void setAngle(double angle) {
        theAngle = angle;
        theThread = new CompassSwingerThread();
        theThread.start();
    }

    /**
     * Returns the currently set angle on the compass.
     * @return the currently set angle on the compass, in radians.
     * @see #setAngle(double)
     */
    public double getAngle() {
        return theAngle;
    }

    /**
     * Returns the preferred size of this component
     * @return the preferred size of this component
     */
    public Dimension getPreferredSize() {
        return new Dimension(300, 300);
    }

    /**
     * Returns the minimum size of this component
     * @return the minimum size of this component
     */
    public Dimension getMinimumSize() {
        return new Dimension(50, 50);
    }


    /**
     * Draws the component on the graphics context.  This is intended to be
     * called from the superclass <code>update(Graphics)</code> method.
     * @param g the graphics context
     */
    protected void paintComponent(Graphics g) {
        super.paintComponent(g);

        int w = getSize().width;
        int h = getSize().height;
        int r = Math.min(w, h) / 2;

        g.translate(w / 2 - r, h / 2 - r);


        // draw dial
        g.setColor(getBackground());
        g.fillRect(0, 0, w, h);
        g.setColor(SystemColor.controlShadow);
        g.fillArc(0, 0, 2 * r - 1, 2 * r - 1, 45, 180);
        g.setColor(SystemColor.controlDkShadow);
        g.fillArc(2, 2, 2 * r - 6, 2 * r - 6, 45, 180);
        g.setColor(SystemColor.controlLtHighlight);
        g.fillArc(0, 0, 2 * r - 1, 2 * r - 1, 225, 180);
        g.setColor(SystemColor.controlHighlight);
        g.fillArc(2, 2, 2 * r - 6, 2 * r - 6, 225, 180);
        g.setColor(SystemColor.control);
        g.fillOval(6, 6, 2 * r - 12, 2 * r - 12);

        int needleLength = r - 10;
        int needleWidth = r / 10;
        int dropShadowHeight = r / 30;
        int fontSize = r / 5;


        // draw letters
        g.setFont(new Font("sans-serif", Font.BOLD, fontSize));
        FontMetrics fm = g.getFontMetrics();
        g.setColor(SystemColor.controlText);
        g.drawString("N", r - fm.stringWidth("N") / 2, fm.getAscent() + 5);
        g.drawString("S", r - fm.stringWidth("S") / 2, 2 * r - 13);
        g.drawString("W", 10, r + fm.getAscent() / 2);
        g.drawString("E", 2 * r - 1 - fm.stringWidth("E") - 8, r + fm.getAscent() / 2);


        // draw ticks
        // I was having some interference between the ticks an the letters.
        // Perhaps someone else can figure out how to make this work.
        //
        g.setColor(SystemColor.controlText);
        for (int i = 0; i < 360; i++) {
            int length;
            int res = i % 90;
            if (res == 0) {
                continue;
            } else if (res == 22 || res == 67) {
                length = r / 10;
                int rs = r - 15;
                g.drawLine(
                        (int) (r + rs * Math.cos(Math.PI * 2 * i / 360)),
                        (int) (r + rs * Math.sin(Math.PI * 2 * i / 360)),
                        (int) (r + (rs - length) * Math.cos(Math.PI * 2 * i / 360)),
                        (int) (r + (rs - length) * Math.sin(Math.PI * 2 * i / 360))
                );
            } else if (res == 47) {
                length = r / 14;
                int rs = r - 20;
                g.fillOval(
                        (int) (r - 5 + rs * Math.cos(Math.PI * 2 * i / 360)),
                        (int) (r - 5 + rs * Math.sin(Math.PI * 2 * i / 360)),
                        10,
                        10
                );
            } else {
                continue;
            }
        }

        // draw needle shadow
        g.setColor(SystemColor.controlShadow);
        Polygon poly = new Polygon();
        poly.addPoint((int) (r + dropShadowHeight + needleLength * Math.sin(thePhysicalAngle)),
                (int) (r + dropShadowHeight - needleLength * Math.cos(thePhysicalAngle)));
        poly.addPoint((int) (r + dropShadowHeight + needleWidth * Math.cos(thePhysicalAngle)),
                (int) (r + dropShadowHeight + needleWidth * Math.sin(thePhysicalAngle)));
        poly.addPoint((int) (r + dropShadowHeight - needleLength * Math.sin(thePhysicalAngle)),
                (int) (r + dropShadowHeight + needleLength * Math.cos(thePhysicalAngle)));
        poly.addPoint((int) (r + dropShadowHeight - needleWidth * Math.cos(thePhysicalAngle)),
                (int) (r + dropShadowHeight - needleWidth * Math.sin(thePhysicalAngle)));
        g.fillPolygon(poly);

        // draw needle
        g.setColor(SystemColor.red);
        poly = new Polygon();
        poly.addPoint((int) (r + needleLength * Math.sin(thePhysicalAngle)), (int) (r - needleLength * Math.cos(thePhysicalAngle)));
        poly.addPoint((int) (r + needleWidth * Math.cos(thePhysicalAngle)), (int) (r + needleWidth * Math.sin(thePhysicalAngle)));
        poly.addPoint((int) (r - needleWidth * Math.cos(thePhysicalAngle)), (int) (r - needleWidth * Math.sin(thePhysicalAngle)));
        g.fillPolygon(poly);
        g.setColor(SystemColor.white);
        poly = new Polygon();
        poly.addPoint((int) (r - needleLength * Math.sin(thePhysicalAngle)), (int) (r + needleLength * Math.cos(thePhysicalAngle)));
        poly.addPoint((int) (r + needleWidth * Math.cos(thePhysicalAngle)), (int) (r + needleWidth * Math.sin(thePhysicalAngle)));
        poly.addPoint((int) (r - needleWidth * Math.cos(thePhysicalAngle)), (int) (r - needleWidth * Math.sin(thePhysicalAngle)));
        g.fillPolygon(poly);
        g.setColor(SystemColor.black);
        g.fillOval(r - needleWidth / 2, r - needleWidth / 2, needleWidth, needleWidth);
    }

    /**
     * An inner class whose instances manage the "swinging" of the compass.
     */
    private class CompassSwingerThread extends Thread {
        public void run() {
            while (theThread == Thread.currentThread()) {

                setName("CompassControl Swinger Thread");

                // calculate how far the needle has to swing
                double ang1 = thePhysicalAngle - Math.floor(thePhysicalAngle / (2 * Math.PI)) * 2 * Math.PI;
                double ang2 = theAngle - Math.floor(theAngle / (2 * Math.PI)) * 2 * Math.PI;
                double dif = ang2 - ang1;
                if (dif < -Math.PI) {
                    dif += 2 * Math.PI;
                } else if (dif > Math.PI) {
                    dif -= 2 * Math.PI;
                }

                // The speed of the "swing" is proportional to the
                // sine of the distance left to go.  This simulates
                // a compass needle moving in a damping fluid.
                double step = Math.sin(dif) * .05;

                // prevents the compass from getting "balanced"
                // at dead center away from the target direction.
                if (step < .001 && step > 0) {
                    step = .001;
                } else if (step >= -.001 && step <= 0) {
                    step = -.001;
                }

                // if we're close enough, put the needle where it should
                // be and exit the loop.
                if (Math.abs(step) > Math.abs(dif)) {
                    thePhysicalAngle = theAngle;
                    repaint();
                    break;
                }
                // otherwise, move the needle slightly.
                else {
                    thePhysicalAngle += step;
                    repaint();
                }

                // This controls the timing and lets other threads
                // have some time.
                try {
                    Thread.sleep(10);
                } catch (InterruptedException e) {
                }
            }
        }
    }
    // Variables declaration - do not modify//GEN-BEGIN:variables

    /** the angle for N */
    public static final double N = 0 * Math.PI / 8;

    /** the angle for NNE */
    public static final double NNE = 1 * Math.PI / 8;

    /** the angle for NE */
    public static final double NE = 2 * Math.PI / 8;

    /** the angle for ENE */
    public static final double ENE = 3 * Math.PI / 8;

    /** the angle for E */
    public static final double E = 4 * Math.PI / 8;

    /** the angle for ESE */
    public static final double ESE = 5 * Math.PI / 8;

    /** the angle for SE */
    public static final double SE = 6 * Math.PI / 8;

    /** the angle for SSE */
    public static final double SSE = 7 * Math.PI / 8;

    /** the angle for S */
    public static final double S = 8 * Math.PI / 8;

    /** the angle for SSW */
    public static final double SSW = 9 * Math.PI / 8;

    /** the angle for SW */
    public static final double SW = 10 * Math.PI / 8;

    /** the angle for WSW */
    public static final double WSW = 11 * Math.PI / 8;

    /** the angle for W */
    public static final double W = 12 * Math.PI / 8;

    /** the angle for WNW */
    public static final double WNW = 13 * Math.PI / 8;

    /** the angle for NW */
    public static final double NW = 14 * Math.PI / 8;

    /** the angle for NNW */
    public static final double NNW = 15 * Math.PI / 8;

    /**
     * the angle set on the compass.  (note:  If the compass is "swinging", this
     * may be different than the apparent angle.
     */
    private double theAngle;

    /**
     * the apparent angle shown on the compass, distinguished from the set value
     * when the compass is swinging.
     */
    private double thePhysicalAngle;

    /**
     * the Thread that manages the "swinging" appearance of the compass.
     */
    private Thread theThread;


    // End of variables declaration//GEN-END:variables

}
