/* -*- tab-width:8; fill-column:85 -*-
 * Copyright (c) 2002-2003 University of Copenhagen.
 * All rights reserved
 *
 * Authors:     Dennis Haney (davh@diku.dk)
 *    Date:     Nov 2002
 *
 * Original copyright:
 *       Copyright  2002 International Business Machines Corporation, 
 *       Massachusetts Institute of Technology, and others. All Rights Reserved. 
 * Originally Licensed under the IBM Public License, see:
 * http://www.opensource.org/licenses/ibmpl.html
 * Previously a part of the bluehoc and blueware simulators
 */

/*********************************************************
 LMP module: This class simulates LMP signaling related to connection
 establishment and QoS negotiation.  It should be very easy to add
 more LMP exchanges.  Since this sits on top of the stop and wait ARQ
 (Link Controller) it provides reliable delivery (at least for a large
 value of tx_thresh).

 Also since this object sits just above the ARQ object in BTNode, it
 seems logical to house the queues containing LMP and L2CAP traffic in
 this layer. Since there is one LMP and one L2CAP queue per slave,
 master will require upto 7 instance of this class if the same class
 is to be used for the slave.

 It also implements those HCI commands which require LMP exchanges
 because the LMP PDUs have to be sent over a reliable channel while
 our BTBaseband class sits below the link controller.
 *********************************************************/

includes bt;
#include "bt_simplequeue.h"

module BTLMPM
{
     provides {
          interface BTHostSig;
          interface BTLMP;
     }
     uses {
          interface BTHost;
          interface BTBaseband;
     }
}

implementation
{
     command void BTLMP.Init(struct LMP* lmp, bool master, amaddr_t am) {
          TRACE_BT(LEVEL_FUNCTION, "%s (%d)\n", __FUNCTION__, __LINE__);
          lmp->am_addr_ = 0;
          lmp->policy_ = (enum link_policy) (EN_SWITCH | EN_HOLD | EN_SNIFF | EN_PARK | EN_CONN);
          lmp->pkts_queued_ = 0;
          simpleq_init(&lmp->lmpq_);
          simpleq_init(&lmp->hostq_);
          lmp->l2capq_.used = 0;
          TRACE_BT(LEVEL_FUNCTION, "%s (%d)\n", __FUNCTION__, __LINE__);
     }

// l2capq_ is now full after receiving P. The next packet will be dropped.
     void queueFull(struct LMP* lmp, struct BTPacket* p) {
          struct hdr_bt* bt = &(p->bt);
          if(lmp->lid_ == bt->lid_) {
               // P is generated by THIS node.
               //So, ignore this advance notification packet and wait until really drop happens
          }
          else {
               signal BTHostSig.queueFull(lmp->lid_, bt->lid_);
          }
     }

     command void BTLMP.dropped(struct LMP* lmp, struct BTPacket* p, bool bFrmQue) {
          struct hdr_bt* bt = &(p->bt);
          struct hdr_cmn* ch = &(p->ch);

          if(bFrmQue) {
               lmp->pkts_queued_ -= SlotSize[bt->type];
          }
          if (bt->ph.l_ch == LMP_CHAN) {
               unsigned char* data = bt->ph.data;
               enum lmp_opcode opcode = (enum lmp_opcode)data[0];
               TRACE_BT(LEVEL_HIGH, "_%d_ DROPPED LMP pkt %d opcode %d\n", call BTBaseband.bd_addr(), ch->uid, opcode);
               return;
          }
          else if(bt->ph.l_ch == HOST_CHAN) {
               signal BTHostSig.dropped(p);
          }
          else {
               call BTHost.droppedApplPacket(p);
               if(bFrmQue && lmp->l2capq_.used == MaxDataQueSize-1)
                    signal BTHostSig.queueFull(lmp->lid_, bt->lid_); // FIXME: to signal or not to signal
          }
     }

     struct BTPacket* l2deque(struct LMP* lmp, int pktSize) {
          struct BTPacket* p;
          struct hdr_bt* bt;
          if (!lmp->l2capq_.used)
               return NULL;
          p = lmp->l2capq_.q[0];
          bt = &(p->bt);
          if(SlotSize[bt->type] <= pktSize) {
               lmp->l2capq_.used--;
               if (lmp->l2capq_.used) //dont do this if there are no elems anyway
                    memmove(lmp->l2capq_.q[0], lmp->l2capq_.q[1], sizeof(struct BTPacket*) * lmp->l2capq_.used);
               TRACE_BT(LEVEL_MED, "_%d_ DQUE %s\n",
                        call BTBaseband.bd_addr(), ptoString(p));
          }
          else {
               p = NULL;
          }
          return p;
     }

     void l2enque(struct LMP* lmp, struct BTPacket* p) {
          TRACE_BT(LEVEL_MED, "_%d_ EQUE %s\n", call BTBaseband.bd_addr(), ptoString(p));

          if(lmp->l2capq_.used == MaxDataQueSize-2) {
               // The next packet will result in drop; let host know about it in advance
               queueFull(lmp, p);
          }
          if (lmp->l2capq_.used == MaxDataQueSize-1) { // FULL
               call BTLMP.dropped(lmp, p, TRUE);
               FREEP(p);
          }
          else {
               lmp->l2capq_.q[lmp->l2capq_.used++] = p;
          }
     }

// L2CAP or LMP packet going down the stack.
     void sendDown(struct LMP* lmp, struct BTPacket* p) {
          struct hdr_bt* bt = &(p->bt);
          unsigned char l_ch = bt->ph.l_ch;
          bt->am_addr = lmp->am_addr_;

          lmp->pkts_queued_ += SlotSize[bt->type]; // do this before enque
          // enque it in the proper queue based on logical channel number
          if (l_ch == LMP_CHAN)
               simpleq_enque(&lmp->lmpq_,p);
          else if (l_ch == HOST_CHAN)
               simpleq_enque(&lmp->hostq_,p);
          else {
               l2enque(lmp,p);
          }
     }

     // Send a LMP packet to the connected part
     void sendLMPCommand(struct LMP* lmp, enum lmp_opcode opcode, unsigned int nArgs, ...) {
          struct BTPacket* p = ALLOCP1(1+(4*nArgs)); // 1byte opcode + N params
          struct hdr_bt* bt = &(p->bt);
          struct hdr_cmn* ch = &(p->ch);
          unsigned char* data;
          unsigned int first,i;
          va_list ap;
          bt->type = BT_DM1;
          ch->size = Payload[bt->type];
          bt->ph.l_ch = LMP_CHAN;

          data = bt->ph.data;
          data[0] = (unsigned char)opcode;
          data++; // incr one byte to start copying params

          va_start(ap, nArgs);

          first = 0;

          for(i = 0; i < nArgs; i++) {
               unsigned int arg = va_arg(ap, unsigned int);
               if(i == 0)
                    first = arg;
               // must be 4 bytes per spec
               memcpy((void*)data, (void*)&arg, 4);
               data += 4;
          }
          va_end(ap);


          TRACE_BT(LEVEL_FUNCTION, "_%d_ sendLMPCommand: %s, nArgs %d, first %d\n",
                   call BTBaseband.bd_addr(), LMPCommandStr[opcode], nArgs,first);
          sendDown(lmp, p);
     }

     command void BTLMP.sendLMPCommand2(struct LMP* lmp, enum lmp_opcode opcode, unsigned int arg1, unsigned int arg2) {
          sendLMPCommand(lmp,opcode,2,arg1,arg2);
     }

     void event_create(event_t* fevent, int mote, long long ftime, struct LMP* lmp);

// When eventvalid is false, it either means the sesssion has ended
// due to the timeout or that a new session has interrupted this
// session.
     command void BTLMP.handle(struct LMP* lmp) {
          if(!lmp->eventvalid) {
               if(call BTBaseband.getSessionInProg() ==LMP_IN_PROG)
                    call BTBaseband.endSession(LMP_IN_PROG);
               lmp->task_ = DIS_ALL;
               return;
          }

          switch(lmp->task_) {
          case EN_CONN:
               switch(lmp->step_) {
               case SEND_CMD:
                    assert(call BTBaseband.getSessionInProg() == LMP_IN_PROG);
                    sendLMPCommand(lmp, LMP_HOST_CONN_REQ, 1, 0);
                    break;
               case RECV_CMD:
                    assert(call BTBaseband.getSessionInProg() == LMP_IN_PROG);
                    sendLMPCommand(lmp, LMP_ACCEPTED , 1, (int)LMP_HOST_CONN_REQ);
                    break;
               case SEND_ACCP:
               case RECV_ACCP:
                    if(call BTBaseband.getSessionInProg() ==LMP_IN_PROG)
                         call BTBaseband.endSession(LMP_IN_PROG);
                    lmp->task_ = DIS_ALL;
                    signal BTHostSig.hciConnectionCompleteEvent(lmp->lid_, call BTBaseband.isMasLink(lmp->lid_));
                    break;
               case SEND_REJ:
               case RECV_REJ:
                    assert(0); //not implemented
               }
               break;
          case EN_HOLD:
               switch(lmp->step_) {
               case SEND_CMD:
               case RECV_CMD:
                    if(!call BTBaseband.isSessionAvail(InvalidLid)) {
                         event_create(&lmp->ev_, NODE_NUM, tos_state.tos_time + 20*SlotTime, lmp);
                         return;
                    }
                    if(lmp->step_ == SEND_CMD) {
                         lmp->mclkn_ = call BTBaseband.mclkn(lmp->lid_) + 2; // Hold starts from the next even (mas) slot
                         sendLMPCommand(lmp, LMP_HOLD_REQ, 2, lmp->intv_, lmp->mclkn_);
                    }
                    else {
                         // need to cater for the case when it is not available
                         sendLMPCommand(lmp, LMP_ACCEPTED, 1, (int)LMP_HOLD_REQ);
                    }
                    break;
               case SEND_ACCP:
               case RECV_ACCP:
                    call BTBaseband.holdLink(lmp->lid_, lmp->intv_, lmp->mclkn_, lmp->step_ == RECV_ACCP);
                    break;
               case SEND_REJ:
               case RECV_REJ:
                    assert(0); //not implemented
               }
               break;
          case EN_SWITCH:
               switch(lmp->step_) {
               case SEND_CMD:
               case RECV_CMD:
                    call BTBaseband.roleChangeInProg(TRUE);
                    if(lmp->step_ == SEND_CMD) {
                         if(!call BTBaseband.isMasLink(lmp->lid_)) {
                              call BTBaseband.sendSlotOffset(lmp->lid_);
                         }
                         sendLMPCommand(lmp, LMP_SWITCH_REQ, 0);
                    }
                    else {
                         if(!call BTBaseband.isMasLink(lmp->lid_)) {
                              call BTBaseband.sendSlotOffset(lmp->lid_);
                         }
                         sendLMPCommand(lmp, LMP_ACCEPTED , 1, LMP_SWITCH_REQ);
                    }
                    break;
               case SEND_ACCP:
               case RECV_ACCP:
                    if(!call BTBaseband.isMasLink(lmp->lid_)) {
                         //if (am_addr_ == 0)
                         //assert(0);
                         if (lmp->am_addr_ == 0)
                              lmp->am_addr_ = lmp->ev_am_addr_;
                         call BTBaseband.switchSlaveRole(lmp->am_addr_);
                    }
                    break;
               case SEND_REJ:
               case RECV_REJ:
                    assert(0); //not implemented
               }
               break;
          default:
               assert(0);
          }
     }

     void event_handle(event_t* fevent, struct TOS_state* state) __attribute__ ((C, spontaneous)) {
          struct LMP* lmp = (struct LMP*)fevent->data;
          call BTLMP.handle(lmp);
     }

     void event_create(event_t* fevent, int mote, long long ftime, struct LMP* lmp) {
          fevent->mote = mote;
          fevent->data = lmp;
          fevent->time = ftime;
          fevent->handle = event_handle;
          fevent->cleanup = NULL;
          fevent->pause = 0;
     }

     command struct BTPacket* BTLMP.send(struct LMP* lmp, int pktSize) {
          struct BTPacket* p = simpleq_deque(&lmp->lmpq_, pktSize);

          if (!p)
               p = simpleq_deque(&lmp->hostq_, pktSize);

          if (!p)
               p = l2deque(lmp, pktSize);

          if(!p) {
               p = call BTHost.send(lmp->lid_);
          }
          else {
               struct hdr_bt* bt = &(p->bt);
               lmp->pkts_queued_ -= SlotSize[bt->type];
               assert(lmp->pkts_queued_ >= 0);
          }
          return p;
     }

     void clear(struct LMP* lmp) {
          struct BTPacket* p = NULL;

          while((p = simpleq_deque(&lmp->lmpq_, MaxBtPktSize)))
               FREEP(p);
          while((p =  simpleq_deque(&lmp->hostq_, MaxBtPktSize)))
               FREEP(p);
          while((p =  l2deque(lmp, MaxBtPktSize)))
               FREEP(p);
          lmp->pkts_queued_ = 0;
     }

     command void BTLMP.linkDestroyed(struct LMP* lmp, bool bMaster) {
          clear(lmp); // clear all the queues.
          signal BTHostSig.hciDisconnectionCompleteEvent(lmp->lid_);
     }

// A new link has been established.
     command void BTLMP.linkEstablished(struct LMP* lmp, bool bMaster) {
          // After a link has established, remain connected for a while by starting a LMP session.
          if(call BTBaseband.getSessionInProg() == NONE_IN_PROG) {
               call BTBaseband.beginSession(LMP_IN_PROG, MinConnSetup, lmp);
               assert(lmp->uid_ <= 0);
          } else if (call BTBaseband.getSessionInProg() != LMP_IN_PROG) {
               event_create(&lmp->ev_, NODE_NUM, tos_state.tos_time + 20*SlotTime, lmp);
               return;
          }
          if(bMaster) {
               // request to create a lmp connection
               lmp->task_ = EN_CONN;
               lmp->step_ = SEND_CMD;
               call BTLMP.handle(lmp);
          }
     }

     command void BTLMP.roleChanged(struct LMP* lmp, bool bMaster) {
          call BTBaseband.roleChangeInProg(FALSE);
          clear(lmp); // clear all the queues.
          signal BTHostSig.hciRoleChangeEvent(bMaster, lmp->lid_);
     }

     // recieved non duplicate packet
     command void BTLMP.recv (struct LMP* lmp, struct BTPacket* p) {
          struct hdr_bt* bt = &(p->bt);
          amaddr_t am_addr = bt->am_addr;

          if (bt->ph.l_ch == LMP_CHAN) {
               unsigned char* data = bt->ph.data;
               // this is totally hacked and will only work on the
               // right endianess, correct here and in
               // BTLMP->sendLMPCommand
               int* params = (int*)(data+1);
               TRACE_BT(LEVEL_HIGH, "_%d_ %s\t AM_ADDR: %-10d %s\n",
                        call BTBaseband.bd_addr(), LMPCommandStr[data[0]], lmp->am_addr_, 
                        ptoString(p));

               switch (data[0]) {
               case LMP_HOST_CONN_REQ:
                    // inform bthost about connection request from remote host
                    // This is a little bit different than other ops since we are allowing the host to decide.
                    assert(am_addr);
                    lmp->am_addr_ = am_addr;
                    signal BTHostSig.hciConnectionRequestEvent(bt->lid_);
                    break;
               case LMP_QOS_REQ:
                    // accept QoS required, could change this later to negotiation
                    sendLMPCommand(lmp, LMP_ACCEPTED, 1, (int)LMP_QOS_REQ);
                    break;
               case LMP_HOLD_REQ:
                    if(lmp->policy_ & EN_HOLD) {
                         lmp->task_ = EN_HOLD;
                         lmp->step_ = RECV_CMD;
                         lmp->intv_ = params[0];
                         lmp->mclkn_ = params[1];
                         call BTLMP.handle(lmp);
                    }
                    else
                         sendLMPCommand(lmp, LMP_NOT_ACCEPTED, 1, (int)LMP_HOLD_REQ);
                    break;
               case LMP_HOLD:
                    // LMP hold received. not implemented!
                    assert(0);
                    break;
               case LMP_SLOT_OFFSET:
                    TRACE_BT(LEVEL_HIGH, "offset %d, bd_addr %d\n", params[0], params[1]);
                    call BTBaseband.recvSlotOffset(
                         //params[0], unused
                         params[1]);
                    break;
               case LMP_SWITCH_REQ:
                    if(lmp->policy_ & EN_SWITCH) {
                         lmp->task_ = EN_SWITCH;
                         lmp->step_ = RECV_CMD;
                         call BTLMP.handle(lmp);
                    }
                    else
                         sendLMPCommand(lmp, LMP_NOT_ACCEPTED, 1, (int)LMP_SWITCH_REQ);
                    break;
               case LMP_DETACH:
                    call BTBaseband.detach(lmp);
                    break;
               case LMP_NOT_ACCEPTED:
                    assert(0);
                    break;
               case LMP_ACCEPTED:
                    TRACE_BT(LEVEL_HIGH, " ACCEPTED PDU: %s\n", LMPCommandStr[params[0]]);

                    switch (params[0]) {
                    case LMP_HOST_CONN_REQ:
                         lmp->step_ = RECV_ACCP;
                         call BTLMP.handle(lmp);
                         break;
                    case LMP_QOS_REQ:
                         // inform bthost that proposed QoS was accepted.
                         assert(0);
                         //signal BTHostSig.hciQoSSetupCompleteEvent(lmp->lid_, qoslm_.packet_type); // TODO: FIXME if QOS
                         break;
                    case LMP_SWITCH_REQ:
                         lmp->ev_am_addr_ = am_addr;
                         lmp->step_ = RECV_ACCP;
                         call BTLMP.handle(lmp);
                         break;
                    case LMP_HOLD_REQ:
                         lmp->step_ = RECV_ACCP;
                         call BTLMP.handle(lmp);
                         break;
                    default:
                         assert(0);
                    }
                    break;
               default:
                    assert(0);
               }
               FREEP(p);
          }
          else if(bt->ph.l_ch == HOST_CHAN) {
               signal BTHostSig.recvPkt(p, bt->lid_);
          }
          else
               signal BTHostSig.recvPktAppl(p);
     }

     // Called with the packet last transmitted and the one just
     // recieved. duplicate test is done later. Packet is free'ed after this call
     command void BTLMP.recvd(struct LMP* lmp, struct BTPacket* p, struct BTPacket* recvdPkt) {
          if(p != NULL) {
               struct hdr_bt* bt = &(p->bt);

               if (bt->ph.l_ch == LMP_CHAN) {
                    unsigned char* data = bt->ph.data;
                    // this is totally hacked and will only work on
                    // the right endianess, correct here and in
                    // BTLMP->sendLMPCommand
                    int* params = (int*)(data+1);

                    switch (data[0]) {
                    case LMP_ACCEPTED:
                         switch (params[0]) {
                         case LMP_HOLD_REQ:
                         case LMP_SWITCH_REQ:
                         case LMP_HOST_CONN_REQ:
                              assert(lmp->task_ != DIS_ALL);
                              lmp->step_ = SEND_ACCP;
                              call BTLMP.handle(lmp);
                              break;
                         default:
                              // who knows, maybe one day we will get free speech back
                              break;
                         }
                         break;
                    }
               }

               if(bt->ph.l_ch != HOST_CHAN && bt->ph.l_ch != LMP_CHAN) {
                    //pass it to L2CAP only if the packet sent was its!
                    call BTHost.recvdAppl(p, lmp->lid_, recvdPkt);
               }
          }
          // Let host know about all the packets being received.
          call BTHost.recvd(p, lmp->lid_, recvdPkt);
     }

     command void BTLMP.transmitted(struct LMP* lmp, struct BTPacket* p) {
          call BTHost.transmitted(p, lmp->lid_);
     }


}
