/*									tab:4
 * "Copyright (c) 2000-2003 The Regents of the University  of California.  
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice, the following
 * two paragraphs and the author appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS."
 *
 * Copyright (c) 2002-2003 Intel Corporation
 * All rights reserved.
 *
 * This file is distributed under the terms in the attached INTEL-LICENSE     
 * file. If you do not find these files, copies can be found by writing to
 * Intel Research Berkeley, 2150 Shattuck Avenue, Suite 1300, Berkeley, CA, 
 * 94704.  Attention:  Intel License Inquiry.
 */

includes DemoCrickets;

/*
 *  Data gather application
 */

module DemoCricketsM {
  provides {
    interface StdControl;
  }
  uses {
    interface StdControl as Sounder;
    interface StdControl as TempStdControl;
    interface StdControl as LightStdControl;

    interface ADC as Light;
    interface ADCControl;
    interface Leds;

    interface Random;

    interface Timer;
    interface Timer as ChirpTimer;
    interface Timer as SendTimer;

    interface Send;
    interface RouteControl;

    interface Epidemic;

    interface AbsoluteTimer;

#ifndef PLATFORM_PC
    interface CC1000Control;
#endif
  }
}

implementation {

  /* Network-Settable Parameters */
  uint8_t darkThreshhold;
  uint32_t timer_rate;
  bool useBeep;
  uint8_t easyMode;
  uint8_t radioPower;

  TOS_Msg gMsgBuffer;
  bool gfSendBusy;

  norace uint16_t gLight;		
  norace uint32_t gTime;
  norace uint32_t gSampleTime;

  uint8_t mySlot;
  uint8_t numSlots;
  uint8_t goodRounds;
  uint8_t goodRound;
  bool myTurn;
  bool chirping;

  /***********************************************************************
   * Initialization 
   ***********************************************************************/
      
  static void initialize() {
    timer_rate = INITIAL_SLOT_LENGTH;
    
    atomic {
      numSlots = INITIAL_NUM_SLOTS;
      mySlot = 0;
      goodRounds = 0;
      goodRound = FALSE;
      myTurn = FALSE;
      darkThreshhold = 0xb0;
      gfSendBusy = FALSE;
      easyMode = TRUE;
      useBeep = FALSE;
    }
    
    call LightStdControl.init();
  }

  task void SendData() {
    CricketsDemoMsg *pReading;
    uint16_t Len;
      
    if (pReading = (CricketsDemoMsg *)call Send.getBuffer(&gMsgBuffer,&Len)) {
      pReading->seq_no++;
      pReading->light = gLight;
      pReading->depth = mySlot;
      atomic {
	  pReading->goodRound = goodRound;
	  pReading->goodRounds = goodRounds;
	  goodRound = FALSE;
	  goodRounds = 0;
      }
/*
      pReading->time = gTime;
      pReading->sampletime = gSampleTime;
*/
      gLight = 0;

      if ((call Send.send(&gMsgBuffer,sizeof(CricketsDemoMsg))) != SUCCESS)
	atomic gfSendBusy = FALSE;
    }
  }
  
  command result_t StdControl.init() {
    initialize();
    return SUCCESS;
  }
  
  command result_t StdControl.start() {
    
    call LightStdControl.start();

    call Epidemic.registerDataItem(EP_RADIO_POWER, 
				   &radioPower, sizeof(radioPower));

    call Epidemic.registerDataItem(EP_DARK_THRESHHOLD,
				   &darkThreshhold, sizeof(darkThreshhold));

    call Epidemic.registerDataItem(EP_TIMER_RATE,
				   &timer_rate, sizeof(timer_rate));

    call Epidemic.registerDataItem(EP_USE_BEEP,
				   &useBeep, sizeof(useBeep));
				   
    call Epidemic.registerDataItem(EP_EASY_MODE,
				   &easyMode, sizeof(easyMode));
    
    if (TOS_LOCAL_ADDRESS != 0) {
      call AbsoluteTimer.setRepeat(timer_rate, 0);
    } else {
      call Leds.greenOn();
    }

    return SUCCESS;
  }

  command result_t StdControl.stop() {
    return call Timer.stop();
  }

  /***********************************************************************
   * Commands and events
   ***********************************************************************/

  event result_t SendTimer.fired() {
    if (!gfSendBusy) {
      gfSendBusy = TRUE;	
      post SendData();
    }
    return SUCCESS;
  }

  event result_t Timer.fired() {
    call Light.getData();
    return SUCCESS;
  }

  event result_t ChirpTimer.fired() {
    if (chirping == FALSE) {
      chirping = TRUE;
      call Sounder.start();
      call ChirpTimer.start(TIMER_ONE_SHOT, 64);
    } else {
      call Sounder.stop();
      chirping = FALSE;
    }
    return SUCCESS;
  }

  event result_t AbsoluteTimer.fired() {
    return SUCCESS;
  }

  event result_t AbsoluteTimer.firedRepeat(tos_time_t t) {
    dbg(DBG_USR1, "@%lld DemoCricketsM: Timer fired at TIME:%d\n",
	tos_state.tos_time/3784, t.low32);

    if (call RouteControl.getParent() != 0xffff)
      call Leds.greenOn();

    call Leds.yellowToggle();
    
    call Timer.start(TIMER_ONE_SHOT, 144);
    
    if ((t.low32 - (mySlot * timer_rate)) 
	% (numSlots * timer_rate) < 32) {
      // it's my turn to sample
      
      dbg(DBG_USR1, "@%d DemoCricketsM: It's my turn to sample\n",
	  t.low32);

      atomic { 
	myTurn = TRUE;
      }
	  
      atomic gSampleTime = t.low32;
      
      call Leds.redOn();
      
      if (useBeep) {
	dbg(DBG_USR1, "BEEP!\n");
	call Sounder.start();
      }
      
    } else if ((t.low32 - ((mySlot+1) * timer_rate)) 
	       % (numSlots * timer_rate) < 32) {
      // it's my turn to send

      call Leds.redOff();
      
      call Sounder.stop();

      atomic {
	myTurn = FALSE;
	gTime = t.low32;
	if (goodRound == TRUE) {
	  call ChirpTimer.start(TIMER_ONE_SHOT, 144); 
	}
	call SendTimer.start(TIMER_ONE_SHOT,
			     call Random.rand() % timer_rate);
      }	  
      
      mySlot = call RouteControl.getDepth();
    }
    
    return SUCCESS;
  }
  
  async event result_t Light.dataReady(uint16_t data) {
    uint16_t myLight;
    
    atomic {
      myLight = data >> 2;
      if (myTurn) {
	gLight = myLight;
      }
      
      if (myLight < darkThreshhold && myTurn) {
	goodRounds++;
	
	if (goodRounds >= numSlots || easyMode == TRUE) {
	  goodRound = TRUE;
	}
      } else if (myLight > darkThreshhold && !myTurn) {
	goodRounds++;
      }
    }
    
    return SUCCESS; 
  }

  event result_t Send.sendDone(TOS_MsgPtr pMsg, result_t success) {
    atomic gfSendBusy = FALSE;
    call Sounder.stop();

    return SUCCESS;
  }


  /* Command interpreter for broadcasts
   *
   */

  event result_t Epidemic.changeHandler(uint8_t channel, uint8_t *pData) {

    if (TOS_LOCAL_ADDRESS == 0) 
      return SUCCESS;

    switch (channel) {

    case EP_RADIO_POWER:
#ifndef PLATFORM_PC
      call CC1000Control.SetRFPower(radioPower);
#endif
      break;

    case EP_TIMER_RATE:
      call AbsoluteTimer.cancel();
      call AbsoluteTimer.setRepeat(timer_rate, 0);
      break;
      
    case EP_DARK_THRESHHOLD:
      break;

    case EP_USE_BEEP:
      break;

    case EP_EASY_MODE:
      break;
    }
    
    return SUCCESS;
  }

/*
  event TOS_MsgPtr Bcast.receive(TOS_MsgPtr pMsg, void* payload, uint16_t payloadLen) {
    SurgeCmdMsg *pCmdMsg = (SurgeCmdMsg *)payload;

    dbg(DBG_USR2, "SurgeM: Bcast  type 0x%02x\n", pCmdMsg->type);

    if (TOS_LOCAL_ADDRESS == 0)
	return pMsg;

    switch (pCmdMsg->type) {

    case SURGE_TYPE_RADIOPOWER:
#ifndef PLATFORM_PC
	call CC1000Control.SetRFPower((uint8_t)pCmdMsg->value);
#endif
	break;

    case SURGE_TYPE_DARKTHRESHHOLD:
	darkThreshhold = pCmdMsg->value;
	break;

    case SURGE_TYPE_SETRATE:
	timer_rate = pCmdMsg->value;
	dbg(DBG_USR2, "SurgeM: set rate %d\n", timer_rate);
	call AbsoluteTimer.cancel();
	call AbsoluteTimer.setRepeat(timer_rate, 0);
	break;
	
    case SURGE_TYPE_BEEP:
	useBeep = TRUE;
	break;
	
    case SURGE_TYPE_BEEPOFF:
	useBeep = FALSE;
	break;

    case SURGE_TYPE_EASYMODE:
	easyMode = TRUE;
	break;
	
    case SURGE_TYPE_HARDMODE:
	easyMode = FALSE;
	break;
    }

    return pMsg;
  }
*/
}











