/*									tab:4
 * "Copyright (c) 2000-2003 The Regents of the University  of California.  
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice, the following
 * two paragraphs and the author appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS."
 *
 * Copyright (c) 2002-2003 Intel Corporation
 * All rights reserved.
 *
 * This file is distributed under the terms in the attached INTEL-LICENSE     
 * file. If you do not find these files, copies can be found by writing to
 * Intel Research Berkeley, 2150 Shattuck Avenue, Suite 1300, Berkeley, CA, 
 * 94704.  Attention:  Intel License Inquiry.
 */

/*
 * Authors:	Gilman Tolle
 *
 */

includes AM;
includes Epidemic;

module EpidemicM {
  provides {
    interface StdControl;
    interface Epidemic[uint32_t comp];
  }
  uses {
    interface StdControl as SubControl;
    interface ReceiveMsg;
    interface SendMsg;
    interface Timer as SendTimer;
    interface Random;
    interface Leds;
  }
}

implementation {

  struct TOS_Msg buf;
  bool currentlySending;
  uint8_t sendEntry;
  uint8_t sendLength;

  enum {
    PRE_SEND,
    POST_SEND,
  };

  typedef struct EpidemicCacheEntry {
    EpidemicMetadata metadata;
    uint8_t          *pData;
    uint8_t          length;
    uint8_t          trickleStage;
    uint16_t         trickleAnnounce;
    uint16_t         trickleCountdown;
    uint8_t          trickleSuppress:1;
    uint8_t          sendToUART:1;
    uint8_t          trickleState:6;
  } __attribute__ ((packed)) EpidemicCacheEntry;

  enum {
    EPIDEMIC_CACHE_ENTRIES = 8,
    TIMER_PERIOD = 64,
    STAGE_LENGTH = 1000,
    MIN_SEND_INTERVAL = 1,
    MAX_SEND_INTERVAL = 64,
  };

  EpidemicCacheEntry epCache[EPIDEMIC_CACHE_ENTRIES];
  EpidemicCacheEntry *signalEntry;

  bool accessingCache = FALSE;

  void printCacheEntry(EpidemicCacheEntry *entry) {
#ifdef PLATFORM_PC
    dbg(DBG_USR1, "@%lld (key=0x%x, seqno=%d), pData=0x%x, length=%d, tstage=%d, tcount=%d, tannounce=%d, ts=%d\n", tos_state.tos_time, entry->metadata.key, entry->metadata.seqno, entry->pData, entry->length, entry->trickleStage, entry->trickleCountdown, entry->trickleAnnounce, entry->trickleState);
#endif
  }

  void printCache() {
    uint8_t i;
    for (i = 0; i < EPIDEMIC_CACHE_ENTRIES; i++) {
      printCacheEntry(&epCache[i]);
    }
  }

  void initEpidemicCache() {
    memset(epCache, sizeof(EpidemicCacheEntry) * EPIDEMIC_CACHE_ENTRIES, 0);
  }

  void trickleSet(EpidemicCacheEntry *epEntry) {
    epEntry->trickleCountdown = epEntry->trickleStage * STAGE_LENGTH;
    epEntry->trickleAnnounce = 
      call Random.rand() % (epEntry->trickleCountdown / 2);
    epEntry->trickleSuppress = FALSE;    
    epEntry->trickleState = PRE_SEND;
  }

  void trickleReset(EpidemicCacheEntry *epEntry) {
    epEntry->trickleStage = 1;
    trickleSet(epEntry);
  }

  EpidemicCacheEntry* findFreeEntry() {
    uint8_t i;
    
    for(i = 0; i < EPIDEMIC_CACHE_ENTRIES-1; i++) {
      if (epCache[i].metadata.key == 0)
	return &epCache[i];
    }
    return NULL;
  }

  EpidemicCacheEntry *findEntry(uint32_t key) {
    uint8_t i;
    
    for(i = 0; i < EPIDEMIC_CACHE_ENTRIES-1; i++) {
      if (epCache[i].metadata.key == key)
	return &epCache[i];
    }
    return NULL;
  }

  command result_t StdControl.init() {
    initEpidemicCache();
    atomic currentlySending = FALSE;
    return call SubControl.init();
  }
  
  command result_t StdControl.start() {
    call SendTimer.start(TIMER_REPEAT, TIMER_PERIOD);
    return call SubControl.start();
  }
  
  command result_t StdControl.stop() {
    return call SubControl.stop();
  }

  command result_t Epidemic.registerDataItem
    [uint32_t comp](uint8_t channel,
			 void *pData, 
			 uint8_t length) {
    /*
      If there is no space in the cache, return FAIL.
      If there is:
        Fill in the source/channel info, set seqno to 0.
	If the ptr is non-null, save the ptr.
    */

    EpidemicCacheEntry *epEntry = findFreeEntry();

    if (epEntry == NULL)
      return FAIL;

    if (length > MAX_DATA_SIZE)
      return FAIL;

    epEntry->metadata.key = MAKE_KEY(comp, channel);
    epEntry->metadata.seqno = 1;
    epEntry->pData = pData;
    epEntry->length = length;

    trickleReset(epEntry);
    printCacheEntry(epEntry);

    return SUCCESS;
  }					    

  command result_t Epidemic.changeValue[uint32_t comp](uint8_t channel) {
    /*
      Look up source and channel in the cache.
      Increment the seqno.
      Reset the trickle timer.
    */

    EpidemicCacheEntry *epEntry = findEntry(MAKE_KEY(comp, channel));
    
    if (epEntry == NULL)
      return FAIL;

    epEntry->metadata.seqno++;
    // Seqno 0 means that the seqno is unknown, and should only be seen
    // in a message coming in from the UART.
    if (epEntry->metadata.seqno == 0)
      epEntry->metadata.seqno++;

    trickleReset(epEntry);
    
    printCache();

    return SUCCESS;
  }

  task void SendDataItemTask() {

    bool result;
    uint8_t i;
    TOS_EpidemicMsg *epMsg = (TOS_EpidemicMsg*) &buf.data;

    if (currentlySending) {
      return;
    }

    i = sendEntry;
  
    memcpy(&epMsg->metadata, &epCache[i].metadata, 
	   sizeof(EpidemicMetadata));
    
    if (epCache[i].pData != NULL) 
      memcpy(epMsg->data, epCache[i].pData, epCache[i].length);
    
    if (epCache[i].sendToUART) {
      /*
	This is a bit of a hack. The previous version sent
	every packet to both the radio and to the UART. That
	may have been causing Multihop to lock up. Now we only
	send to the UART when we need to. 
      */
      result = call SendMsg.send(TOS_UART_ADDR, 
				 sizeof(TOS_EpidemicMsg), &buf);
    } else {
      result = call SendMsg.send(TOS_BCAST_ADDR, 
				 offsetof(TOS_EpidemicMsg,data) + 
				 epCache[i].length,
				 &buf);
    }
    
    if (result == SUCCESS) {
      epCache[i].sendToUART = FALSE;
      atomic currentlySending = TRUE;
      epCache[i].trickleState = POST_SEND;
      printCacheEntry(&epCache[i]);
    }
  }

  event result_t SendTimer.fired() {

    /*
      For each non-null entry in the cache:
        Decrement the countdown timer.
	If the countdown timer goes below the send time and we haven't
	sent yet:
	  If we are not suppressed, send.
	  Wait until the next timer interval to examine the next entry.
	If the countdown timer goes to 0, double the trickle timer.
    */
	
    uint8_t i;

    for(i = 0; i < EPIDEMIC_CACHE_ENTRIES; i++) {

      if (epCache[i].metadata.key == 0)
	continue;

      // Decrement the counter
      if (epCache[i].trickleCountdown > TIMER_PERIOD)
	epCache[i].trickleCountdown -= TIMER_PERIOD;

      if (epCache[i].trickleCountdown < epCache[i].trickleAnnounce &&
	  epCache[i].trickleState == PRE_SEND) {
	
	// If it crosses below the announcement time and we haven't announced
	// yet, announce unless suppressed.

	if (epCache[i].trickleSuppress) {
	    epCache[i].trickleState = POST_SEND;

	} else {
	  
	  /* we're clear to send. send, double, and wait. */
	  atomic sendEntry = i;
	  post SendDataItemTask();

	  break;
	}

      } else if (epCache[i].trickleCountdown <= TIMER_PERIOD) {
	// If it's close to 0, double the trickle timer	
	if (epCache[i].trickleStage < MAX_SEND_INTERVAL)
	  epCache[i].trickleStage *= 2;
	
	trickleSet(&epCache[i]);
      }
    }
    return SUCCESS;
  }

  task void SignalTask() {

    signal Epidemic.changeHandler[KEY_TO_COMPONENT(signalEntry->metadata.key)]
      (KEY_TO_PARAM(signalEntry->metadata.key), signalEntry->pData);

  }

  event TOS_MsgPtr ReceiveMsg.receive(TOS_MsgPtr pMsg) {
    /*
      If it's an EpidemicMsg:
        Look up source and channel in the cache.
	If the seqno is newer than what we have:
	  Update the seqno.
	  If the ptr is non-null:
	    Copy the new value to the data holder.
	  Reset the Trickle timer
	  signal Epidemic.changeHandler()
	If the seqno is the same, set the suppression flag
	If the seqno is older:
	  Reset the Trickle timer to update that node quickly
    */

    TOS_EpidemicMsg *epMsg = (TOS_EpidemicMsg*) pMsg->data;

    EpidemicCacheEntry *epEntry = 
      findEntry(epMsg->metadata.key);

    if (epEntry == NULL)
      return pMsg;

    if (epMsg->metadata.seqno == 0) {
      /* It's coming in from outside, and does not know the seqno. 
	 Send back a message. */
      epEntry->sendToUART = TRUE;
      trickleReset(epEntry);
      return pMsg;
    }

    if ((epMsg->metadata.seqno - epEntry->metadata.seqno) > 0) {

      /* my entry is older. update and rebroadcast. */

      epEntry->metadata.seqno = epMsg->metadata.seqno;

      if (epEntry->pData != NULL)
	memcpy(epEntry->pData, epMsg->data, epEntry->length);

      trickleReset(epEntry);

      atomic signalEntry = epEntry;
      post SignalTask();

    } else if ((epMsg->metadata.seqno - epEntry->metadata.seqno) == 0) {
      
      /* my entry is equal. suppress. */

      epEntry->trickleSuppress = TRUE;

    } else {

      /* my entry is newer. rebroadcast. */

      trickleReset(epEntry);
    }
 

    return pMsg;
  }

  default event result_t Epidemic.changeHandler
    [uint32_t comp](uint8_t channel, 
			 uint8_t *pData) {
    /* Do nothing */
    return SUCCESS;
  }

  event result_t SendMsg.sendDone(TOS_MsgPtr pMsg, 
				  result_t success) {
    atomic currentlySending = FALSE;

    return SUCCESS;
  }
}









