/*
 * Copyright (c) 2003, Vanderbilt University
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice, the following
 * two paragraphs and the author appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE VANDERBILT UNIVERSITY BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE VANDERBILT
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE VANDERBILT UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE VANDERBILT UNIVERSITY HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
/*
 * AcousticLocalization.java
 *
 * Created on August 18, 2003, 1:37 PM
 */

package net.tinyos.acousticlocalization;

/**
 *
 * @author  sallai
 */
import net.tinyos.mcenter.*;
import net.tinyos.packet.*;
import java.util.*;
import javax.swing.*;
import java.io.*;
import java.awt.event.*;
import java.awt.*;

public class AcousticLocalization extends MessageCenterInternalFrame implements PacketListenerIF {
    public static final byte RANGING_APPID = 0x02;
    public static final int FLOODROUTING_AMGROUP = 130;
    public static final int START_RANGING = 1;
    public static final int STOP_RANGING = 0;
    public static final byte TSNEG_APPID = 0x15;
    public static final int START_TSNEG = 1;
    public static final int STOP_TSNEG = 0;
    public static final byte GRADIENTPOLICY_APPID = -127; // 0x81
    public static final int SET_ROOT = 2;
    
    public RangingTableModel tableModel = new AcousticLocalization.RangingTableModel();
    private File currentDir = new File("/");
    private Vector localizedMotes = null;
    private ArrayList lastMeasurements = new ArrayList();
    
    /** Creates new form AcousticLocalization */
    public AcousticLocalization() {
        super("Acoustic Localization");
        
        initComponents();
        
        // register for all floodrouting messages
        SerialConnector.instance().registerPacketListener(this, SerialConnector.GET_ALL_MESSAGES);
        // get rid of the toolbar message after 5 seconds
        ActionListener taskPerformer = new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
                statusLine.setText("");
            }
        };
        new javax.swing.Timer(10000, taskPerformer).start();
        
    }
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    private void initComponents() {//GEN-BEGIN:initComponents
        java.awt.GridBagConstraints gridBagConstraints;

        statusLine = new javax.swing.JLabel();
        tabbedPane = new javax.swing.JTabbedPane();
        jPanel1 = new javax.swing.JPanel();
        tableScrollPane = new javax.swing.JScrollPane();
        measurementsTable = new javax.swing.JTable();
        rangingControljPanel = new javax.swing.JPanel();
        startRangingButton = new javax.swing.JButton();
        stopRangingButton = new javax.swing.JButton();
        stopTSNegButton = new javax.swing.JButton();
        startTSNegButton = new javax.swing.JButton();
        fileControlPanel = new javax.swing.JPanel();
        saveButton = new javax.swing.JButton();
        fileNameField = new javax.swing.JTextField();
        resetButton = new javax.swing.JButton();
        loadButton = new javax.swing.JButton();
        fileSelectButton = new javax.swing.JButton();
        initPanel = new javax.swing.JPanel();
        setRootMoteButton = new javax.swing.JButton();
        rootMoteIDField = new javax.swing.JTextField();
        jLabel1 = new javax.swing.JLabel();
        jPanel2 = new javax.swing.JPanel();
        jPanel3 = new javax.swing.JPanel();
        jScrollPane1 = new javax.swing.JScrollPane();
        fixPositionsTextPane = new javax.swing.JTextPane();
        jPanel4 = new javax.swing.JPanel();
        locViewPanel = new AcousticLocalization.LocViewPanel();
        jPanel5 = new javax.swing.JPanel();
        localizeButton = new javax.swing.JButton();
        writeOutputButton = new javax.swing.JButton();
        posFileNameField = new javax.swing.JTextField();
        posFileSelectButton = new javax.swing.JButton();

        getContentPane().add(statusLine, java.awt.BorderLayout.SOUTH);

        jPanel1.setLayout(new java.awt.GridBagLayout());

        tableScrollPane.setBorder(new javax.swing.border.TitledBorder("Ranging measurements"));
        measurementsTable.setModel(tableModel);
        tableScrollPane.setViewportView(measurementsTable);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        jPanel1.add(tableScrollPane, gridBagConstraints);

        rangingControljPanel.setLayout(new java.awt.GridBagLayout());

        rangingControljPanel.setBorder(new javax.swing.border.TitledBorder("Ranging control"));
        startRangingButton.setText("Start Ranging");
        startRangingButton.setToolTipText("Send start message to motes");
        startRangingButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                startRangingButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        rangingControljPanel.add(startRangingButton, gridBagConstraints);

        stopRangingButton.setText("Stop Ranging");
        stopRangingButton.setToolTipText("Send stop message to motes");
        stopRangingButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                stopRangingButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        rangingControljPanel.add(stopRangingButton, gridBagConstraints);

        stopTSNegButton.setText("Stop Time Slot Negotiation");
        stopTSNegButton.setToolTipText("Send stop message to motes");
        stopTSNegButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                stopTSNegButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        rangingControljPanel.add(stopTSNegButton, gridBagConstraints);

        startTSNegButton.setText("Start Time Slot Negotiation");
        startTSNegButton.setToolTipText("Send start message to motes");
        startTSNegButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                startTSNegButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        rangingControljPanel.add(startTSNegButton, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridheight = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 0.5;
        jPanel1.add(rangingControljPanel, gridBagConstraints);

        fileControlPanel.setLayout(new java.awt.GridBagLayout());

        fileControlPanel.setBorder(new javax.swing.border.TitledBorder("File options"));
        saveButton.setText("Save");
        saveButton.setToolTipText("Load ranging data to the file specified below");
        saveButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                saveButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        fileControlPanel.add(saveButton, gridBagConstraints);

        fileNameField.setText("ranging.txt");
        fileNameField.setToolTipText("File name");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        fileControlPanel.add(fileNameField, gridBagConstraints);

        resetButton.setText("Reset");
        resetButton.setToolTipText("Reset application (clears ranging measurements, known positions and localization results)");
        resetButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                resetButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        fileControlPanel.add(resetButton, gridBagConstraints);

        loadButton.setText("Load");
        loadButton.setToolTipText("Load ranging data from the file specified below");
        loadButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                loadButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        fileControlPanel.add(loadButton, gridBagConstraints);

        fileSelectButton.setText("..");
        fileSelectButton.setToolTipText("Select file");
        fileSelectButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                fileSelectButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.VERTICAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        fileControlPanel.add(fileSelectButton, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 0.5;
        jPanel1.add(fileControlPanel, gridBagConstraints);

        initPanel.setLayout(new java.awt.GridBagLayout());

        initPanel.setBorder(new javax.swing.border.TitledBorder("Routing options"));
        setRootMoteButton.setText("Initialize FloodRouting");
        setRootMoteButton.setToolTipText("Initialize FloodRouting by setting the root node");
        setRootMoteButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                setRootMoteButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        initPanel.add(setRootMoteButton, gridBagConstraints);

        rootMoteIDField.setText("100");
        rootMoteIDField.setToolTipText("ID of the mote that is closest to to the base station");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        initPanel.add(rootMoteIDField, gridBagConstraints);

        jLabel1.setText("ID of root mote:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        initPanel.add(jLabel1, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 0.5;
        jPanel1.add(initPanel, gridBagConstraints);

        tabbedPane.addTab("Ranging", jPanel1);

        jPanel2.setLayout(new java.awt.GridBagLayout());

        jPanel3.setLayout(new java.awt.GridBagLayout());

        jPanel3.setBorder(new javax.swing.border.TitledBorder("Search space and fixed mote positions"));
        fixPositionsTextPane.setBorder(null);
        fixPositionsTextPane.setText("-100.0\t-100.0\t-100.0\n 100.0\t 100.0\t 100.0\n");
        jScrollPane1.setViewportView(fixPositionsTextPane);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        jPanel3.add(jScrollPane1, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        jPanel2.add(jPanel3, gridBagConstraints);

        tabbedPane.addTab("Fix positions", jPanel2);

        jPanel4.setLayout(new java.awt.GridBagLayout());

        locViewPanel.setBorder(new javax.swing.border.TitledBorder("Map of calculated positions"));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        jPanel4.add(locViewPanel, gridBagConstraints);

        jPanel5.setLayout(new java.awt.GridBagLayout());

        jPanel5.setBorder(new javax.swing.border.TitledBorder("Localization"));
        localizeButton.setText("Calculate Positions");
        localizeButton.setToolTipText("Calculate unknown mote positions");
        localizeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                localizeButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 50);
        jPanel5.add(localizeButton, gridBagConstraints);

        writeOutputButton.setText("Write positions to file");
        writeOutputButton.setToolTipText("Write calculated positions to file");
        writeOutputButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                writeOutputButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        jPanel5.add(writeOutputButton, gridBagConstraints);

        posFileNameField.setText("positions.txt");
        posFileNameField.setToolTipText("Name of the positions file");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        jPanel5.add(posFileNameField, gridBagConstraints);

        posFileSelectButton.setText("..");
        posFileSelectButton.setToolTipText("Select file");
        posFileSelectButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                posFileSelectButtonActionPerformed(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 5, 5);
        jPanel5.add(posFileSelectButton, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        jPanel4.add(jPanel5, gridBagConstraints);

        tabbedPane.addTab("Localization", jPanel4);

        getContentPane().add(tabbedPane, java.awt.BorderLayout.CENTER);

        pack();
    }//GEN-END:initComponents
    
    private void setRootMoteButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_setRootMoteButtonActionPerformed
        
        RemoteControl.sendInteger(Integer.decode(rootMoteIDField.getText()).intValue(), GRADIENTPOLICY_APPID, SET_ROOT);
        
    }//GEN-LAST:event_setRootMoteButtonActionPerformed
    
    private void writeOutputButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_writeOutputButtonActionPerformed
        if(localizedMotes == null) {
            statusLine.setText("There are no caclulated positions");
            return;
        }
        
        try {
            FileWriter o = new FileWriter(posFileNameField.getText());
            
            Iterator i = localizedMotes.iterator();
            while(i.hasNext()) {
                MoteInfo m = (MoteInfo)i.next();
                o.write("pos\t"+m.getMoteID()+"\t"+m.getPosition().toString2()+"\n");
            }
            
            o.close();
        } catch(Exception e) {
            //System.err.println("IO Exception: "+e.toString());
            //e.printStackTrace(System.err);
            statusLine.setText("Error writing file ("+e.getMessage()+")");
            return;
        }
        
    }//GEN-LAST:event_writeOutputButtonActionPerformed
    
    private void posFileSelectButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_posFileSelectButtonActionPerformed
        JFileChooser chooser = new JFileChooser(currentDir);
        int returnVal = chooser.showOpenDialog(this.getParent());
        if(returnVal == JFileChooser.APPROVE_OPTION) {
            posFileNameField.setText(chooser.getSelectedFile().getAbsolutePath());
            currentDir = chooser.getCurrentDirectory();
        }
        
    }//GEN-LAST:event_posFileSelectButtonActionPerformed
    
    private void fileSelectButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_fileSelectButtonActionPerformed
        JFileChooser chooser = new JFileChooser(currentDir);
        int returnVal = chooser.showOpenDialog(this.getParent());
        if(returnVal == JFileChooser.APPROVE_OPTION) {
            fileNameField.setText(chooser.getSelectedFile().getAbsolutePath());
            currentDir = chooser.getCurrentDirectory();
        }
    }//GEN-LAST:event_fileSelectButtonActionPerformed
    
    private void loadButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_loadButtonActionPerformed
        DistanceMeasurementFileReader dmfr = new DistanceMeasurementFileReader();
        
        reset();
        
        // read file
        try {
            dmfr.readFromFile(fileNameField.getText());
        } catch(IOException e) {
            //System.err.println("IO Exception: "+e.toString());
            //e.printStackTrace(System.err);
            statusLine.setText("Error reading file ("+e.getMessage()+")");
            return;
        }
        catch(FileFormatException e) {
            //System.err.println("FileFormatException: "+e.toString());
            //e.printStackTrace(System.err);
            statusLine.setText("Error reading file ("+e.getMessage()+")");
            return;
        }
        
        // fill measurement table
        for(int i=0; i<dmfr.moteNum; i++)
            for(int j=0; j<dmfr.moteNum; j++) {
                for(int k=0; k<dmfr.distances[i][j].size(); k++) {
                    
                    if(dmfr.distances[i][j].get(k)>=0) {
                        int idi = ((MoteInfo)dmfr.motes.get(i)).getMoteID();
                        int idj = ((MoteInfo)dmfr.motes.get(j)).getMoteID();
                        tableModel.addMeasurement(new AcousticLocalization.Measurement(idi, idj, dmfr.distances[i][j].get(k)));
                        // System.out.println("LOAD: Adding measurement: "+idi+" "+idj+" "+dmfr.distances[i][j].get(k));
                    }
                }
            }
        
        // fill search space and known mote positions
        StringBuffer sb = new StringBuffer();
        try {
            BufferedReader br = new BufferedReader(new FileReader(fileNameField.getText()));
            String line;
            while((line = br.readLine())!=null) {
                boolean isDistLine = false;
                StringTokenizer st = new StringTokenizer(line);
                if(st.hasMoreTokens())
                    if(st.nextToken().equalsIgnoreCase("dist") )
                        isDistLine = true;
                if(!isDistLine) sb.append(line+"\n");
                
                
            }
            fixPositionsTextPane.setText(sb.toString());
        } catch(IOException e) {
            //System.err.println("IO Exception: "+e.toString());
            //e.printStackTrace(System.err);
            statusLine.setText("Error reading file ("+e.getMessage()+")");
            return;
        }
/*
        // fill search space and known mote positions
        StringBuffer sb = new StringBuffer();
        sb.append(dmfr.min.getX()+"\t"+dmfr.min.getY()+"\t"+dmfr.min.getZ()+"\n");
        sb.append(dmfr.max.getX()+"\t"+dmfr.max.getY()+"\t"+dmfr.max.getZ()+"\n");
 
        Iterator i = dmfr.motes.iterator();
        while(i.hasNext()) {
            MoteInfo m = (MoteInfo)i.next();
 
            if(m.hasFixedCoord()) {
                // sb.append("pos\t"+m.moteID+"\t"+m.centerPosition.getX()+"\t"+m.centerPosition.getY()+"\t"+m.centerPosition.getZ()+"\n");
                sb.append("pos\t"+m.getMoteID()+"\t");
 
                if(m.getFixedMask()[0]) sb.append(m.getPosition().getX()+"\t");
                else sb.append("x\t");
 
                if(m.getFixedMask()[1]) sb.append(m.getPosition().getY()+"\t");
                else sb.append("y\t");
 
                if(m.getFixedMask()[2]) sb.append(m.getPosition().getZ()+"\t");
                else sb.append("z\t");
 
                sb.append("\n");
                // System.out.println("LOAD: Adding position: "+"pos\t"+m.moteID+"\t"+m.centerPosition.getX()+"\t"+m.centerPosition.getY()+"\t"+m.centerPosition.getZ()+"\n");
            }
        }
 
        fixPositionsTextPane.setText(sb.toString());
 */
    }//GEN-LAST:event_loadButtonActionPerformed
    
    private void saveButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_saveButtonActionPerformed
        try {
            //FileOutputStream o = new FileOutputStream(fileNameField.getText());
            FileWriter w = new FileWriter(fileNameField.getText());
            w.write(fixPositionsTextPane.getText()+"\n");
            Iterator i = tableModel.measurements.iterator();
            while(i.hasNext()) {
                AcousticLocalization.Measurement m = (AcousticLocalization.Measurement)i.next();
                w.write("dist\t"+m.actuator+"\t"+m.sensor+"\t"+m.distance+"\n");
            }
            
            w.close();
        } catch(FileNotFoundException e) {
            //System.err.println("FileNotFoundException: "+e.getMessage());
            //e.printStackTrace(System.err);
            statusLine.setText("Cannot write file ("+e.getMessage()+")");
        }
        catch(IOException e) {
            //System.err.println("IOException: "+e.getMessage());
            //e.printStackTrace(System.err);
            statusLine.setText("Cannot write file ("+e.getMessage()+")");
        }
        
    }//GEN-LAST:event_saveButtonActionPerformed
    
    public void reset() {
        tableModel.measurements = new ArrayList();
        tableModel.fireTableDataChanged();
        fixPositionsTextPane.setText("");
        fixPositionsTextPane.invalidate();
        this.localizedMotes = null;
        ((LocViewPanel)locViewPanel).reset();
        locViewPanel.repaint();
    }
    
    private void resetButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_resetButtonActionPerformed
        reset();
    }//GEN-LAST:event_resetButtonActionPerformed
    
    public LocalizationProblem solve(LocalizationProblem localizationProblem, int steps, boolean randomizedLocations) {
        
        LocalizationProblem lp = (LocalizationProblem)localizationProblem.clone();
        
        int n = steps;
        double s = 0.1;
        if (randomizedLocations) lp.random();
        for( int i=0; i<n; ++i ) {
            double learningrate = s * Math.pow( 0.0001/s, i/(double)n );
            
            lp.step3( learningrate );
            //lp.step1( learningrate );
            lp.step2( learningrate );
            
            if( i%100 == 0 ) {
                ((AcousticLocalization.LocViewPanel)locViewPanel).setMotes(lp.motes);
                ((AcousticLocalization.LocViewPanel)locViewPanel).setDistances(lp.distances);
                ((AcousticLocalization.LocViewPanel)locViewPanel).setSearchSpace(lp.min, lp.max);
                ((AcousticLocalization.LocViewPanel)locViewPanel).repaint();
                
                //System.out.print( 100*i/n + "%\t" );
                //System.out.println(lp.calcDistanceError2());
            }
        }
        
        return lp;
    }
    
    public LocalizationProblem getBestSolution(LocalizationProblem lp, int trials, int iterations) {
        LocalizationProblem lp_trial;
        LocalizationProblem lp_bestTrial = lp;
        double error;
        double bestError = lp.calcDistanceError2();
        
        
        for(int i=0; i<trials; i++) {
            lp_trial = solve(lp, iterations, true);
            error = lp_trial.calcDistanceError2();
            
            if(error<bestError) {
                bestError = error;
                lp_bestTrial = lp_trial;
            }
        }
        return lp_bestTrial;
    }
    
    private void localizeButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_localizeButtonActionPerformed
        // do nothing if no measurements
        if(tableModel.measurements.size() == 0) {
            statusLine.setText("Cannot calculate positions because no measurement is present");
            return;
        }
        
        // build input string for DistanceMeasurementFileReader
        StringBuffer localizationInputSB = new StringBuffer(fixPositionsTextPane.getText());
        localizationInputSB.append("\n");
        Iterator i = tableModel.measurements.iterator();
        while(i.hasNext()) {
            AcousticLocalization.Measurement m = (AcousticLocalization.Measurement)i.next();
            localizationInputSB.append("dist\t"+m.actuator+"\t"+m.sensor+"\t"+m.distance+"\n");
        }
        
        // parse the string
        DistanceMeasurementFileReader dmfr = new DistanceMeasurementFileReader();
        try {
            dmfr.readFromString(localizationInputSB.toString());
        } catch(IOException e) {
            //System.err.println("IO Exception: "+e.toString());
            //e.printStackTrace(System.err);
            statusLine.setText("Error in search space/fixed positions text ("+e.getMessage()+")");
            return;
        }
        catch(FileFormatException e) {
            //System.err.println("FileFormatException: "+e.toString());
            //e.printStackTrace(System.err);
            statusLine.setText("Data format error in search space/fixed positions text ("+e.getMessage()+")");
            return;
        }
        
        
        DistanceMeasurements dm = dmfr.getDistanceMeasurements();
        // do nothing if moteNum<3
        if(dm.moteNum<3) {
            statusLine.setText("Cannot calculate positions because less than 3 motes are specified");
            return;
        }
        
        // solve it
        LocalizationProblem lp = new LocalizationProblem();
        
        lp.motes = dm.motes;
        lp.moteNum = dm.moteNum;
        lp.distances = dm.getSymmetricDistancesArray(); // take average
        lp.min = dmfr.getMin();
        lp.max = dmfr.getMax();
        lp.removeMotesWithLessThan3Neighbor();
       
        lp = getBestSolution(lp,5,20000);
        localizedMotes = lp.motes;
                
    }//GEN-LAST:event_localizeButtonActionPerformed
     
    private void stopTSNegButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_stopTSNegButtonActionPerformed
        stopTSNeg();
    }//GEN-LAST:event_stopTSNegButtonActionPerformed
    
    private void startTSNegButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_startTSNegButtonActionPerformed
        startTSNeg();
    }//GEN-LAST:event_startTSNegButtonActionPerformed
    
    private void stopRangingButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_stopRangingButtonActionPerformed
        stopRanging();
    }//GEN-LAST:event_stopRangingButtonActionPerformed
    
    private void startRangingButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_startRangingButtonActionPerformed
        startRanging();
    }//GEN-LAST:event_startRangingButtonActionPerformed
    
    
    public void packetReceived(byte[] packet) {
        //        System.out.print("Packet received: ");
        //        for(int i=0; i<packet.length; i++) System.out.print(((short)packet[i] & 0xff) +" ");
        //        System.out.println();
        
        if((packet[2] & 0xff) != 130) return; // return if it's not a floodrouting message
        
        int packetIndex = 5;
        byte appID = (byte)(packet[packetIndex++] & 0xFF);
        int location = (short)((packet[packetIndex++] & 0xFF)
        + ((packet[packetIndex++] & 0xFF) << 8)) & 0xFFFF;
        
        if(appID == RANGING_APPID) { // it's a ranging message
            int actuatorID = (short)((packet[packetIndex++] & 0xFF)
            + ((packet[packetIndex++] & 0xFF) << 8)) & 0xFFFF;
            int sensorID =   (short)((packet[packetIndex++] & 0xFF)
            + ((packet[packetIndex++] & 0xFF) << 8)) & 0xFFFF;
            short distance =   (short)((packet[packetIndex++] & 0xFF)
            + ((packet[packetIndex++] & 0xFF) << 8));
            //System.out.println("Ranging message received: actuator="+actuatorID+" sensor="+sensorID+" distance="+distance);
            if(distance > 0) { // check if valid measurement (filter out -1's)
                AcousticLocalization.Measurement m = new AcousticLocalization.Measurement(actuatorID, sensorID, distance);
                if(!lastMeasurements.contains(m)) { // filter out multiple instances of the same message
                    tableModel.addMeasurement(m);
                    lastMeasurements.add(m);
                    if(lastMeasurements.size()>10) lastMeasurements.remove(0);
                }
            }
        }
    }
    
    public void startRanging() {
        RemoteControl.sendInteger(RemoteControl.BROADCAST, RANGING_APPID, START_RANGING);
        
    }
    
    public void stopRanging() {
        RemoteControl.sendInteger(RemoteControl.BROADCAST, RANGING_APPID, STOP_RANGING);
        
    }
    
    public void startTSNeg() {
        RemoteControl.sendInteger(RemoteControl.BROADCAST, TSNEG_APPID, START_TSNEG);
        
    }
    
    public void stopTSNeg() {
        RemoteControl.sendInteger(RemoteControl.BROADCAST, TSNEG_APPID, STOP_TSNEG);
        
    }
    
    private class RangingTableModel extends javax.swing.table.AbstractTableModel {
        public ArrayList measurements  = new ArrayList();
        
        public void addMeasurement(Measurement m) {
            measurements.add(m);
            this.fireTableDataChanged();
        }
        
        /** Returns the number of columns in the model. A
         * <code>JTable</code> uses this method to determine how many columns it
         * should create and display by default.
         *
         * @return the number of columns in the model
         * @see #getRowCount
         */
        public int getColumnCount() {
            return 3;
        }
        
        /** Returns the number of rows in the model. A
         * <code>JTable</code> uses this method to determine how many rows it
         * should display.  This method should be quick, as it
         * is called frequently during rendering.
         *
         * @return the number of rows in the model
         * @see #getColumnCount
         */
        public int getRowCount() {
            return measurements.size();
        }
        
        /** Returns the value for the cell at <code>columnIndex</code> and
         * <code>rowIndex</code>.
         *
         * @param   rowIndex    the row whose value is to be queried
         * @param   columnIndex     the column whose value is to be queried
         * @return  the value Object at the specified cell
         */
        public Object getValueAt(int rowIndex, int columnIndex) {
            Measurement m = (Measurement)measurements.toArray()[rowIndex];
            switch(columnIndex) {
                case 0: return new Integer(m.actuator);
                
                case 1: return new Integer(m.sensor);
                
                case 2: return new Double(m.distance);
            }
            return null;
        }
        
        public String getColumnName(int column) {
            switch(column) {
                case 0: return "Actuator";
                
                case 1: return "Sensor";
                
                case 2: return "Distance";
            }
            return "";
        }
    }
    
    public class Measurement {
        
        int actuator;
        int sensor;
        double distance;
        
        public Measurement() {}
        public Measurement(int s, int a, double d) {
            this.sensor = s;
            this.actuator = a;
            this.distance = d;
        }
        
        public boolean equals(Object o) {
            Measurement m = (Measurement)o;
            if(m.actuator == actuator && m.sensor == sensor && m.distance == distance) return true;
            else return false;
        }
    }
    
    public class LocViewPanel extends JPanel {
        private Vector motes;
        private Position min, max;
        private double[][] distances;
        
        public void setMotes(Vector motes) {
            this.motes = motes;
        }
        
        public void setDistances(double[][] d) {
            this.distances = d;
        }
        
        public void setSearchSpace(Position min, Position max) {
            this.min = min;
            this.max = max;
            
            if(min.coord[0]>max.coord[0]) {
                double tmp = max.coord[0];
                max.coord[0] = min.coord[0];
                min.coord[0] =tmp;
            }
            if(min.coord[1]>max.coord[1]) {
                double tmp = max.coord[1];
                max.coord[1] = min.coord[1];
                min.coord[1] =tmp;
            }
        }
        
        public void reset() {
            this.motes = null;
            this.min = new Position();
            this.max = new Position();
            this.distances = null;
        }
        
        protected void paintComponent(Graphics g) {
            super.paintComponent(g);
            int i,j;
            double sfactor;
            
            if(motes == null || distances == null) return;
            
            Dimension dim = getSize();
            
            Image offscreen = createImage(dim.width,dim.height);
            
            Graphics b = offscreen.getGraphics();
            
            // set scaling
            sfactor = dim.width/(max.coord[0] - min.coord[0]);
            if(dim.height/(max.coord[1] - min.coord[1])<sfactor) sfactor = dim.height/(max.coord[1] - min.coord[1]);
            
            
            // draw edges
            b.setColor( Color.black );
            for( i=0; i<motes.size()-1; ++i ) {
                MoteInfo m1 = (MoteInfo)motes.get(i);
                for( j=i+1; j<motes.size(); ++j ) {
                    MoteInfo m2 = (MoteInfo)motes.get(j);
                    if( distances[i][j] > 0 ) {
                        int x1 = (int)((m1.getPosition().getX() - min.getX()) * sfactor);
                        int y1 = dim.height - (int)((m1.getPosition().getY() - min.getY()) * sfactor);
                        int x2 = (int)((m2.getPosition().getX() - min.getX()) * sfactor);
                        int y2 = dim.height - (int)((m2.getPosition().getY() - min.getY()) * sfactor);
                        System.out.println("dist\t"+m1.getMoteID()+"\t"+m2.getMoteID()+"\t"+ distances[i][j]);
                        b.drawLine( x1, y1, x2, y2 );
                    }
                }
            }
            
            
            // draw positions
            b.setColor( Color.blue );
            for( i=0; i<motes.size(); ++i ) {
                MoteInfo m1 = (MoteInfo)motes.get(i);
                int x = (int)((m1.getPosition().getX() - min.getX()) * sfactor);
                int y = dim.height - (int)((m1.getPosition().getY() - min.getY()) * sfactor);
                
                int r = 5;
                if( m1.isFixed() )
                    r = 10;
                
                b.drawOval( x-r/2, y-r/2, r, r );
                b.drawString( Integer.toString(m1.getMoteID()), x+2+r/2, y+2+r/2 );
            }
            
            g.drawImage(offscreen,0,0,this);
        }
    }
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JTable measurementsTable;
    private javax.swing.JPanel jPanel4;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JButton stopTSNegButton;
    private javax.swing.JPanel jPanel3;
    private javax.swing.JScrollPane tableScrollPane;
    private javax.swing.JButton fileSelectButton;
    private javax.swing.JButton startRangingButton;
    private javax.swing.JTextPane fixPositionsTextPane;
    private javax.swing.JButton loadButton;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JButton stopRangingButton;
    private javax.swing.JPanel jPanel5;
    private javax.swing.JPanel locViewPanel;
    private javax.swing.JTextField rootMoteIDField;
    private javax.swing.JButton localizeButton;
    private javax.swing.JLabel statusLine;
    private javax.swing.JPanel fileControlPanel;
    private javax.swing.JTextField fileNameField;
    private javax.swing.JPanel initPanel;
    private javax.swing.JTabbedPane tabbedPane;
    private javax.swing.JPanel rangingControljPanel;
    private javax.swing.JButton saveButton;
    private javax.swing.JButton posFileSelectButton;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JButton resetButton;
    private javax.swing.JButton writeOutputButton;
    private javax.swing.JTextField posFileNameField;
    private javax.swing.JButton startTSNegButton;
    private javax.swing.JButton setRootMoteButton;
    // End of variables declaration//GEN-END:variables
    
}
