/*									tab:4
 *  IMPORTANT: READ BEFORE DOWNLOADING, COPYING, INSTALLING OR USING.  By
 *  downloading, copying, installing or using the software you agree to
 *  this license.  If you do not agree to this license, do not download,
 *  install, copy or use the software.
 *
 *  Intel Open Source License 
 *
 *  Copyright (c) 2002 Intel Corporation 
 *  All rights reserved. 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are
 *  met:
 * 
 *	Redistributions of source code must retain the above copyright
 *  notice, this list of conditions and the following disclaimer.
 *	Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *      Neither the name of the Intel Corporation nor the names of its
 *  contributors may be used to endorse or promote products derived from
 *  this software without specific prior written permission.
 *  
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *  PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE INTEL OR ITS
 *  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 *  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  @author Leah Fera, Martin Turon, Jaidev Prabhu
 *
 *  $Id$ 
 */


/******************************************************************************
 *
 *    - Tests the MDA300 general prototyping card 
 *       (see Crossbow MTS Series User Manual)
 *    -  Read and control all MDA300 signals:
 *      ADC0, ADC1, ADC2, ADC3,...ADC11 inputs, DIO 0-5, 
 *      counter, battery, humidity, temp
 *-----------------------------------------------------------------------------
 * Output results through mica2 uart and radio. 
 * Use xlisten.exe program to view data from either port:
 *  uart: mount mica2 on mib510 with MDA300 
 *              (must be connected or now data is read)
 *        connect serial cable to PC
 *        run xlisten.exe at 57600 baud
 *  radio: run mica2 with MDA300, 
 *         run another mica2 with TOSBASE
 *         run xlisten.exe at 56K baud
 * LED: the led will be green if the MDA300 is connected to the mica2 and 
 *      the program is running (and sending out packets).  Otherwise it is red.
 *-----------------------------------------------------------------------------
 * Data packet structure:
 * 
 * PACKET #1 (of 4)
 * ----------------
 *  msg->data[0] : sensor id, MDA300 = 0x81
 *  msg->data[1] : packet number = 1
 *  msg->data[2] : node id
 *  msg->data[3] : reserved
 *  msg->data[4,5] : analog adc data Ch.0
 *  msg->data[6,7] : analog adc data Ch.1
 *  msg->data[8,9] : analog adc data Ch.2
 *  msg->data[10,11] : analog adc data Ch.3
 *  msg->data[12,13] : analog adc data Ch.4
 *  msg->data[14,15] : analog adc data Ch.5
 *  msg->data[16,17] : analog adc data Ch.6
 * 
 * PACKET #2 (of 4)
 * ----------------
 *  msg->data[0] : sensor id, MDA300 = 0x81
 *  msg->data[1] : packet number = 2
 *  msg->data[2] : node id
 *  msg->data[3] : reserved
 *  msg->data[4,5] : analog adc data Ch.7
 *  msg->data[6,7] : analog adc data Ch.8
 *  msg->data[8,9] : analog adc data Ch.9
 *  msg->data[10,11] : analog adc data Ch.10
 *  msg->data[12,13] : analog adc data Ch.11
 *  msg->data[14,15] : analog adc data Ch.12
 *  msg->data[16,17] : analog adc data Ch.13
 *
 * 
 * PACKET #3 (of 4)
 * ----------------
 *  msg->data[0] : sensor id, MDA300 = 0x81
 *  msg->data[1] : packet number = 3
 *  msg->data[2] : node id
 *  msg->data[3] : reserved
 *  msg->data[4,5] : digital data Ch.0
 *  msg->data[6,7] : digital data Ch.1
 *  msg->data[8,9] : digital data Ch.2
 *  msg->data[10,11] : digital data Ch.3
 *  msg->data[12,13] : digital data Ch.4
 *  msg->data[14,15] : digital data Ch.5
 *
 * PACKET #4 (of 4)
 * ----------------
 *  msg->data[0] : sensor id, MDA300 = 0x81
 *  msg->data[1] : packet number = 4
 *  msg->data[2] : node id
 *  msg->data[3] : reserved
 *  msg->data[4,5] : batt
 *  msg->data[6,7] : hum
 *  msg->data[8,9] : temp
 *  msg->data[10,11] : counter
 *  msg->data[14] : msg4_status (debug)
 * 
 ***************************************************************************/

// include sensorboard.h definitions from tos/mda300 directory
includes sensorboard;

module XSensorMDA300M
{
  
    provides interface StdControl;
  
    uses {
	interface Leds;

	//Sampler Communication
	interface StdControl as SamplerControl;
	interface Sample;

	//UART communication
	interface StdControl as UARTControl;
	interface BareSendMsg as UARTSend;
	interface ReceiveMsg as UARTReceive;

	//RF communication
	interface StdControl as CommControl;
	interface SendMsg as SendMsg;
	interface ReceiveMsg as ReceiveMsg;
    
	//Timer
	interface Timer;
    
	//relays
	interface Relay as relay_normally_closed;
	interface Relay as relay_normally_open;   
    
	//support for plug and play
	command result_t PlugPlay();
    }
}


implementation
{ 	
#define ANALOG_SAMPLING_TIME    90
#define DIGITAL_SAMPLING_TIME  100
#define MISC_SAMPLING_TIME     110

#define ANALOG_SEND_FLAG  1
#define DIGITAL_SEND_FLAG 1
#define MISC_SEND_FLAG    1
#define ERR_SEND_FLAG     1

#define PACKET1_FULL	0x7F
#define PACKET2_FULL	0x7F
#define PACKET3_FULL	0x3F
#define PACKET4_FULL	0x0F

#define MSG_LEN  29   // excludes TOS header, but includes xbow header
    
    enum {
	PENDING = 0,
	NO_MSG = 1
    };        

    enum {
	MDA300_PACKET1 = 1,
	MDA300_PACKET2 = 2,
	MDA300_PACKET3 = 3,
	MDA300_PACKET4 = 4,
	MDA300_ERR_PACKET = 0xf8	
    };

    enum {
	SENSOR_ID = 0,
	PACKET_ID, 
	NODE_ID,
	RESERVED,
	DATA_START
    } XPacketDataEnum;
	
    /* Messages Buffers */	
	
    TOS_Msg packet[5];
    TOS_Msg uart_send_buffer, radio_send_buffer;    
    TOS_MsgPtr uart_msg_ptr, radio_msg_ptr;
    
    TOS_Msg errMsg_uart, errMsg_radio;
     
    uint16_t errMsg_status;

    uint8_t pkt_send_order[4];
    uint8_t next_packet;
    uint8_t packet_ready;
    bool    sending_packet;
    uint8_t msg_status[5], pkt_full[5];
    char test;

    int8_t record[25];
 
/****************************************************************************
 * Initialize the component. Initialize Leds
 *
 ****************************************************************************/
    command result_t StdControl.init() {
        
	uint8_t i;

	call Leds.init();
        
	atomic {
	    errMsg_status=0;    
	    uart_msg_ptr = &uart_send_buffer;
	    radio_msg_ptr = &radio_send_buffer;

	    pkt_send_order[0] = 1;
	    pkt_send_order[1] = 2;
	    pkt_send_order[2] = 3;
	    pkt_send_order[3] = 4;

	    packet_ready = 0;
	    next_packet = 0;
	    sending_packet = FALSE;
	}
	for (i=1; i<=4; i++) 
	    msg_status[i] = 0;
	pkt_full[1] = PACKET1_FULL;
	pkt_full[2] = PACKET2_FULL;
	pkt_full[3] = PACKET3_FULL;
	pkt_full[4] = PACKET4_FULL;
        
	return rcombine(call SamplerControl.init(), call CommControl.init());
    }

/** Sends a plain text error string using the text_msg board type. */
    task void send_uart_err_msg(){
	uint8_t i;
	char *errMsg = "mda300 not found";

	errMsg_status = 1 && ERR_SEND_FLAG;  
	if (!errMsg_status) return;

	errMsg_uart.data[SENSOR_ID] = SENSOR_BOARD_ID;
	errMsg_uart.data[PACKET_ID] = MDA300_ERR_PACKET;
	errMsg_uart.data[NODE_ID] = TOS_LOCAL_ADDRESS;
	errMsg_uart.addr = TOS_UART_ADDR;
	errMsg_uart.type = 0;
	errMsg_uart.length = MSG_LEN; //TOSH_DATA_LENGTH;
	errMsg_uart.group = TOS_AM_GROUP;  
	i = 0;
	// Copy error string
	while ((*errMsg) && (i <= MSG_LEN-1)) {
	    errMsg_uart.data[DATA_START + i] = errMsg[i];
	    i++;
	}

	// Copy over uart packet to radio packet (identical)
	for (i = 0; i <= MSG_LEN-1; i++) errMsg_radio.data[i] = errMsg_uart.data[i];

	call UARTSend.send(&errMsg_uart);
    }
 
/****************************************************************************
 * Start the component. Start the clock. Setup timer and sampling
 *
 ****************************************************************************/
    command result_t StdControl.start() {
        
	call SamplerControl.start();
        
	call CommControl.start();
        
	if(call PlugPlay())
	{
            
	    call Timer.start(TIMER_REPEAT, 3000);
            
            
	    //channel parameteres are irrelevent
            
	    record[14] = call Sample.getSample(0,TEMPERATURE,MISC_SAMPLING_TIME,SAMPLER_DEFAULT);
            
	    record[15] = call Sample.getSample(0,HUMIDITY,MISC_SAMPLING_TIME,SAMPLER_DEFAULT);
            
	    record[16] = call Sample.getSample(0, BATTERY,MISC_SAMPLING_TIME,SAMPLER_DEFAULT);

            
	    //start sampling  channels. Channels 7-10 with averaging since they are more percise.channels 3-6 make active excitation    
	    record[0] = call Sample.getSample(0,ANALOG,ANALOG_SAMPLING_TIME,SAMPLER_DEFAULT | EXCITATION_33);

	    record[1] = call Sample.getSample(1,ANALOG,ANALOG_SAMPLING_TIME,SAMPLER_DEFAULT );
            
	    record[2] = call Sample.getSample(2,ANALOG,ANALOG_SAMPLING_TIME,SAMPLER_DEFAULT);
            
	    record[3] = call Sample.getSample(3,ANALOG,ANALOG_SAMPLING_TIME,SAMPLER_DEFAULT | EXCITATION_33 | DELAY_BEFORE_MEASUREMENT);
            
	    record[4] = call Sample.getSample(4,ANALOG,ANALOG_SAMPLING_TIME,SAMPLER_DEFAULT);
            
	    record[5] = call Sample.getSample(5,ANALOG,ANALOG_SAMPLING_TIME,SAMPLER_DEFAULT);
            
	    record[6] = call Sample.getSample(6,ANALOG,ANALOG_SAMPLING_TIME,SAMPLER_DEFAULT);
            
	    record[7] = call Sample.getSample(7,ANALOG,ANALOG_SAMPLING_TIME,AVERAGE_FOUR | EXCITATION_25);
            
	    record[8] = call Sample.getSample(8,ANALOG,ANALOG_SAMPLING_TIME,AVERAGE_FOUR | EXCITATION_25);
            
	    record[9] = call Sample.getSample(9,ANALOG,ANALOG_SAMPLING_TIME,AVERAGE_FOUR | EXCITATION_25);
            
	    record[10] = call Sample.getSample(10,ANALOG,ANALOG_SAMPLING_TIME,AVERAGE_FOUR | EXCITATION_25);
         
	    record[11] = call Sample.getSample(11,ANALOG,ANALOG_SAMPLING_TIME,SAMPLER_DEFAULT);
            
	    record[12] = call Sample.getSample(12,ANALOG,ANALOG_SAMPLING_TIME,SAMPLER_DEFAULT);
            
	    record[13] = call Sample.getSample(13,ANALOG,ANALOG_SAMPLING_TIME,SAMPLER_DEFAULT | EXCITATION_50 | EXCITATION_ALWAYS_ON);                                
                        
            
	    //digital chennels as accumulative counter                
            
	    record[17] = call Sample.getSample(0,DIGITAL,DIGITAL_SAMPLING_TIME,RESET_ZERO_AFTER_READ | FALLING_EDGE);
            
	    record[18] = call Sample.getSample(1,DIGITAL,DIGITAL_SAMPLING_TIME,RISING_EDGE | EVENT);
            
	    record[19] = call Sample.getSample(2,DIGITAL,DIGITAL_SAMPLING_TIME,SAMPLER_DEFAULT | EVENT);
            
	    record[20] = call Sample.getSample(3,DIGITAL,DIGITAL_SAMPLING_TIME,FALLING_EDGE);
            
	    record[21] = call Sample.getSample(4,DIGITAL,DIGITAL_SAMPLING_TIME,RISING_EDGE);
            
	    record[22] = call Sample.getSample(5,DIGITAL,DIGITAL_SAMPLING_TIME,RISING_EDGE | EEPROM_TOTALIZER);                                
            
	    //counter channels for frequency measurement, will reset to zero.
            
	    record[23] = call Sample.getSample(0, COUNTER,MISC_SAMPLING_TIME,RESET_ZERO_AFTER_READ | RISING_EDGE);
	    call Leds.greenOn();          
	}
        
	else {
	    post send_uart_err_msg();
	    call Leds.redOn();
	}
        
	return SUCCESS;
    
    }
    
/****************************************************************************
 * Stop the component.
 *
 ****************************************************************************/
 
    command result_t StdControl.stop() {
        
 	call SamplerControl.stop();
    
 	return SUCCESS;
    
    }


/****************************************************************************
 * Task to uart as message
 *
 ****************************************************************************/
    task void send_uart_msg(){
	uint8_t i;

	atomic sending_packet = TRUE;
	uart_msg_ptr->addr = TOS_UART_ADDR;
	uart_msg_ptr->type = 0;
	uart_msg_ptr->length = MSG_LEN; 
	uart_msg_ptr->group = TOS_AM_GROUP;  
	uart_msg_ptr->data[SENSOR_ID] = SENSOR_BOARD_ID;
	uart_msg_ptr->data[PACKET_ID] = next_packet;
	uart_msg_ptr->data[NODE_ID] = TOS_LOCAL_ADDRESS;
	for (i = 4; i <= MSG_LEN-1; i++) 
	    uart_msg_ptr->data[i] = packet[next_packet].data[i];
	call UARTSend.send(uart_msg_ptr);

    }

/****************************************************************************
 * Task to transmit radio message
 * NOTE that data payload was already copied from the corresponding UART packet
 ****************************************************************************/
    task void send_radio_msg() 
	{
	    uint8_t i;
	
	    radio_msg_ptr->addr = TOS_BCAST_ADDR;
	    radio_msg_ptr->type = 0;
	    radio_msg_ptr->length = MSG_LEN; //TOSH_DATA_LENGTH;
	    radio_msg_ptr->group = TOS_AM_GROUP;
	    radio_msg_ptr->data[SENSOR_ID] = SENSOR_BOARD_ID;
	    radio_msg_ptr->data[PACKET_ID] = next_packet;
	    radio_msg_ptr->data[NODE_ID] = TOS_LOCAL_ADDRESS;
	    for (i = 4; i <= MSG_LEN-1; i++) 
		    radio_msg_ptr->data[i] = packet[next_packet].data[i];
	    call SendMsg.send(TOS_BCAST_ADDR, MSG_LEN, radio_msg_ptr); 

	}


/****************************************************************************
 * Uart msg xmitted. 
 * Transmit same msg over radio
 ****************************************************************************/
    event result_t UARTSend.sendDone(TOS_MsgPtr msg, result_t success) {
    
	uart_msg_ptr = msg;
	call Leds.yellowOn();

 	post send_radio_msg(); 
    	return SUCCESS;
    }
  
/****************************************************************************
 * Radio msg xmitted. 
 ****************************************************************************/
    event result_t SendMsg.sendDone(TOS_MsgPtr msg, result_t success) {
    
	radio_msg_ptr = msg;
	call Leds.yellowOff();
	
        // mark that this packet has been sent
	atomic {
	    sending_packet = FALSE;
	    packet_ready &= ~(1 << (next_packet - 1));
	}
	return SUCCESS;
    }

/****************************************************************************
 * Radio msg rcvd. 
 * This app doesn't respond to any incoming radio msg
 * Just return
 ****************************************************************************/
    event TOS_MsgPtr ReceiveMsg.receive(TOS_MsgPtr data)  {
	return data;
    }
 
/****************************************************************************
 * Uart msg rcvd. 
 * This app doesn't respond to any incoming uart msg
 * Just return
 ****************************************************************************/
    event TOS_MsgPtr UARTReceive.receive(TOS_MsgPtr data) {
	return data;
    }
 
/**
 * Handle a single dataReady event for all MDA300 data types. 
 * 
 * @author    Leah Fera, Martin Turon
 *
 * @version   2004/3/17       leahfera    Intial revision
 * @n         2004/4/1        mturon      Improved state machine
 */
    event result_t 
	Sample.dataReady(uint8_t channel,uint8_t channelType,uint16_t data)
	{          
	    uint8_t i;

	    switch (channelType) {
		case ANALOG:              
		    switch (channel) {		  
			// MSG 1 : first part of analog channels (0-6)
			case 0:
			    packet[1].data[DATA_START+0]=data & 0xff;
			    packet[1].data[DATA_START+1]=(data >> 8) & 0xff;
			    atomic {msg_status[1] |=0x01;}
			    break;

			case 1:   
			    packet[1].data[DATA_START+2]=data & 0xff;
			    packet[1].data[DATA_START+3]=(data >> 8) & 0xff;
			    atomic {msg_status[1] |=0x02;}
			    break;
             
			case 2:
			    packet[1].data[DATA_START+4]=data & 0xff;
			    packet[1].data[DATA_START+5]=(data >> 8) & 0xff;
			    atomic {msg_status[1] |=0x04;}
			    break;
              
			case 3:
			    packet[1].data[DATA_START+6]=data & 0xff;
			    packet[1].data[DATA_START+7]=(data >> 8) & 0xff;
			    atomic {msg_status[1] |=0x08;}
			    break;
              
			case 4:
			    packet[1].data[DATA_START+8]=data & 0xff;
			    packet[1].data[DATA_START+9]=(data >> 8) & 0xff;
			    atomic {msg_status[1] |=0x10;}
			    break;
              
			case 5:
			    packet[1].data[DATA_START+10]=data & 0xff;
			    packet[1].data[DATA_START+11]=(data >> 8) & 0xff;
			    atomic {msg_status[1] |=0x20;}
			    break;
              
			case 6:
			    packet[1].data[DATA_START+12]=data & 0xff;
			    packet[1].data[DATA_START+13]=(data >> 8) & 0xff;
			    atomic {msg_status[1]|=0x40;}
			    break;

			    // MSG 2 : second part of analog channels (7-13)
                	case 7:
			    packet[2].data[DATA_START+0]=data & 0xff;
			    packet[2].data[DATA_START+1]=(data >> 8) & 0xff;
			    atomic {msg_status[2]|=0x01;}
			    break;

                	case 8:
			    packet[2].data[DATA_START+2]=data & 0xff;
			    packet[2].data[DATA_START+3]=(data >> 8) & 0xff;
			    atomic {msg_status[2]|=0x02;}
			    break;
              
			case 9:
			    packet[2].data[DATA_START+4]=data & 0xff;
			    packet[2].data[DATA_START+5]=(data >> 8) & 0xff;
			    atomic {msg_status[2]|=0x04;}
			    break;
              
			case 10:
			    packet[2].data[DATA_START+6]=data & 0xff;
			    packet[2].data[DATA_START+7]=(data >> 8) & 0xff;
			    atomic {msg_status[2]|=0x08;}
			    break;
             
			case 11:
			    packet[2].data[DATA_START+8]=data & 0xff;
			    packet[2].data[DATA_START+9]=(data >> 8) & 0xff;
			    atomic {msg_status[2]|=0x10;}
			    break;
              
			case 12:
			    packet[2].data[DATA_START+10]=data & 0xff;
			    packet[2].data[DATA_START+11]=(data >> 8) & 0xff;
			    atomic {msg_status[2]|=0x20;}
			    break;
              
			case 13:
			    packet[2].data[DATA_START+12]=data & 0xff;
			    packet[2].data[DATA_START+13]=(data >> 8) & 0xff;
			    atomic {msg_status[2]|=0x40;}
			    break;
              
			default:
			    break;
		    }  // case ANALOG (channel) 
		    break;
          
		case DIGITAL:
		    switch (channel) {             
			case 0:
			    packet[3].data[2]=data & 0xff;
			    packet[3].data[3]=(data >> 8) & 0xff;
			    atomic {msg_status[3]|=0x01;}
			    break;
              
			case 1:
			    packet[3].data[4]=data & 0xff;
			    packet[3].data[5]=(data >> 8) & 0xff;
			    atomic {msg_status[3]|=0x02;}
			    break;
            
			case 2:
			    packet[3].data[6]=data & 0xff;
			    packet[3].data[7]=(data >> 8) & 0xff;
			    atomic {msg_status[3]|=0x04;}
			    break;
              
			case 3:
			    packet[3].data[8]=data & 0xff;
			    packet[3].data[9]=(data >> 8) & 0xff;
			    atomic {msg_status[3]|=0x08;}
			    break;
              
			case 4:
			    packet[3].data[10]=data & 0xff;
			    packet[3].data[11]=(data >> 8) & 0xff;
			    atomic {msg_status[3]|=0x10;}
			    break;
              
			case 5:
			    packet[3].data[12]=data & 0xff;
			    packet[3].data[13]=(data >> 8) & 0xff;
			    atomic {msg_status[3]|=0x20;}
			    break;
              
			default:
			    break;
		    }  // case DIGITAL (channel)
		    break;

		case BATTERY:            
		    packet[4].data[4]=data & 0xff;            
		    packet[4].data[5]=(data >> 8) & 0xff;            
		    atomic {msg_status[4]|=0x01;}
		    break;
          
		case HUMIDITY:            
		    packet[4].data[6]=data & 0xff;            
		    packet[4].data[7]=(data >> 8) & 0xff;            
		    atomic {msg_status[4]|=0x02;}
		    break;
                    
		case TEMPERATURE:          
		    packet[4].data[8]=data & 0xff;           
		    packet[4].data[9]=(data >> 8) & 0xff;
		    atomic {msg_status[4]|=0x04;}
		    break;

		case COUNTER:
		    packet[4].data[10]=data & 0xff;
		    packet[4].data[11]=(data >> 8) & 0xff;
		    atomic {msg_status[4]|=0x08;}
		    break;  

		default:
		    break;

	    }  // switch (channelType) 

	    atomic {            
		for (i=1; i<=4; i++) {
		    if (sending_packet)
                        // avoid posting uart_send-Task while one is in process
			break; 

		    next_packet = pkt_send_order[0];

		    pkt_send_order[0] = pkt_send_order[1];
		    pkt_send_order[1] = pkt_send_order[2];
		    pkt_send_order[2] = pkt_send_order[3];
		    pkt_send_order[3] = next_packet;

		    if (msg_status[next_packet] == pkt_full[next_packet]) {
			msg_status[next_packet] = 0;
			packet_ready |= 1 << (next_packet - 1);
			post send_uart_msg();
			break;
		    } 
		}
	    }
          
	    return SUCCESS;      
	}
  
/****************************************************************************
 * Timer Fired - 
 *
 ****************************************************************************/
    event result_t Timer.fired() {
      
 	if (test != 0)  {
	    test=0;
	    call relay_normally_closed.toggle();
 	}
	else  {
	    test=1;
	    call relay_normally_open.toggle();
 	}

	return SUCCESS;
  
    }

}
