/*									tab:4
 *  IMPORTANT: READ BEFORE DOWNLOADING, COPYING, INSTALLING OR USING.  By
 *  downloading, copying, installing or using the software you agree to
 *  this license.  If you do not agree to this license, do not download,
 *  install, copy or use the software.
 *
 *  Intel Open Source License 
 *
 *  Copyright (c) 2002 Intel Corporation 
 *  All rights reserved. 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are
 *  met:
 * 
 *	Redistributions of source code must retain the above copyright
 *  notice, this list of conditions and the following disclaimer.
 *	Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *      Neither the name of the Intel Corporation nor the names of its
 *  contributors may be used to endorse or promote products derived from
 *  this software without specific prior written permission.
 *  
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *  PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE INTEL OR ITS
 *  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 *  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * 
 * $Id$
 */

/******************************************************************************
 *    -Measures mica2dot battery voltage using the on-board voltage reference. 
 *     As the battery voltage changes the Atmega ADC's full scale decreases. By
 *     measuring a known voltage reference the battery voltage can be computed.
 *
 *
 *    - Measure the mica2dot thermistor resistance.
 *      The thermistor and voltage reference share the same adc channel (ADC1).
 *      They have indiviual on/off controls:
 *      thermistor:  PW6 = lo => on; PW6 = hi => off
 *      voltage ref: PW7 = lo => on; PW7 = hi => off
 *      They cannot both be turned on together.
 *      Give ~msec of settling time after applying power before making a measurement.
 *
 *    -Tests the MDA500 general prototyping card (see Crossbow MTS Series User Manaul)
 *     Read and control all MDA500 signals:
 *     - reads ADC2,ADC3,...ADC7 inputs
 *     - toggles the following MDA500 I/O lines at a 1Hz rate:
 *       THERM_PWR (GPS_EN on mica2dot pcb), PWM1B,INT1,INT0,PW0,PW1
 *-----------------------------------------------------------------------------
 * Output results through mica2dot uart and radio. 
 * Use Console.exe program to view data from either port:
 *  uart: mount mica2dot on mib510 with or without MDA500
 *        connect serial cable to PC
 *        run xlisten.exe at 19200 baud
 *  radio: run mica2dot with or withouth MDA500, 
 *         run mica2 with TOSBASE
 *         run xlisten.exe at 56K baud
 *-----------------------------------------------------------------------------
 * Data packet structure  :
 *  msg->data[0] : sensor id, MDA500 = 0x1
 *  msg->data[1] : packet id
 *  msg->data[2] : node id
 *  msg->data[3] : reserved
 *  msg->data[4,5] : battery adc data
 *  msg->data[6,7] : thermistor adc data
 *  msg->data[8,9] : adc2 data
 *  msg->data[10,11] : adc3 data
 *  msg->data[12,13] : adc4 data
 *	msg->data[14,15] : adc5 data
 *	msg->data[16,17] : adc6 data
 *  msg->data[18,19] : adc7 data
 * 
 *------------------------------------------------------------------------------
 * Mica2Dot:
 * The thermistor and voltage reference share the same adc channel (ADC1).
 * They have indiviual on/off controls:
 *  thermistor:  PW6 = lo => on; PW6 = hi => off
 *  voltage ref: PW7 = lo => on; PW7 = hi => off
 *  They cannot both be turned on together.
 *  Give ~msec of settling time after applying power before making a measurement.
 *
 *****************************************************************************/

module TestSensorM {
  provides {
    interface StdControl;
  }
  uses {
  //  interface Clock;
	interface ADC as ADCBATT;
	interface ADC as ADC2;
    interface ADC as ADC3;
    interface ADC as ADC4;
    interface ADC as ADC5;
    interface ADC as ADC6;
    interface ADC as ADC7;
    

  	interface ADCControl;
    interface Timer;
	interface Leds;

	interface StdControl as UARTControl;
    interface BareSendMsg as UARTSend;
    interface ReceiveMsg as UARTReceive;

	interface StdControl as RadioControl;
    interface BareSendMsg as RadioSend;
    interface ReceiveMsg as RadioReceive;
  }
}

implementation {

  enum { SENSOR_ID = 0,PACKET_ID, NODE_ID=2,RSVD, 
         VREF=4, 
         THERM = 6,
		 ADC2_D = 8,
		 ADC3_D = 10,
		 ADC4_D = 12,
		 ADC5_D = 14,
		 ADC6_D = 16,
		 ADC7_D = 18};
  
  enum {STATE0 = 0, STATE1,STATE2};
  #define MSG_LEN  29 

   TOS_Msg msg_buf_uart, msg_buf_radio;
   TOS_MsgPtr msg_uart,msg_radio;
   bool bGetVoltRef;
   bool bIOon;
   bool bLedOn;
   uint8_t state;

/****************************************************************************
 * Task to xmit radio message
 *
 ****************************************************************************/
   task void send_radio_msg(){
    msg_radio->data[SENSOR_ID] = SENSOR_BOARD_ID;
    msg_radio->data[PACKET_ID] = 1;     // Only one packet for MDA500
    msg_radio->data[NODE_ID] = TOS_LOCAL_ADDRESS;
    msg_radio->data[RSVD] = 0;
    msg_radio->addr = TOS_BCAST_ADDR;
	msg_radio->type = 0;
	msg_radio->length = MSG_LEN;
	msg_radio->group = TOS_AM_GROUP;
    call RadioSend.send(msg_radio);
    return;
  }
/****************************************************************************
 * Task to uart as message
 *
 ****************************************************************************/
   task void send_uart_msg(){
   uint8_t i;

    msg_uart->data[SENSOR_ID] = SENSOR_BOARD_ID;
    msg_uart->data[PACKET_ID] = 1;      // Only one packet for MDA500
    msg_uart->data[NODE_ID] = TOS_LOCAL_ADDRESS;
    msg_uart->data[RSVD] = 0;
    msg_uart->addr = TOS_UART_ADDR;
    msg_uart->type = 0;
    msg_uart->length = MSG_LEN;
    msg_uart->group = TOS_AM_GROUP;  
    for (i = 0; i <= MSG_LEN-1; i++) msg_radio->data[i] = msg_uart->data[i];
    call UARTSend.send(msg_uart);
    return;
  }

 /****************************************************************************
 * Initialize the component. Initialize ADCControl, Leds
 *
 ****************************************************************************/
  command result_t StdControl.init() {
    atomic{
	bGetVoltRef = TRUE;
	bIOon = FALSE;
	bLedOn = FALSE;
        msg_uart = &msg_buf_uart;
	msg_radio = &msg_buf_radio;
        state = STATE0;
    };
// set atmega pin directions for mda500	
	MAKE_THERM_OUTPUT();             //enable thermistor power pin as output
    MAKE_BAT_MONITOR_OUTPUT();       //enable voltage ref power pin as output
    MAKE_INT0_OUTPUT();
	MAKE_INT1_OUTPUT();
	MAKE_PWO_OUTPUT();
	MAKE_PW1_OUTPUT();
	MAKE_PWM1B_OUTPUT();
	MAKE_GPS_ENA_OUTPUT();

	call ADCControl.init();
    call Leds.init();
	call UARTControl.init();
    call RadioControl.init();
    call Leds.init();
   	return SUCCESS;

  }
 /****************************************************************************
 * Start the component. Start the clock.
 *
 ****************************************************************************/
  command result_t StdControl.start(){
    call UARTControl.start();
    call RadioControl.start();
	call Timer.start(TIMER_REPEAT, 500);
    return SUCCESS;	
  }
 /****************************************************************************
 * Stop the component.
 *
 ****************************************************************************/
  command result_t StdControl.stop() {
    call UARTControl.stop();
    call RadioControl.stop();
    return SUCCESS;    
  }
/****************************************************************************
 * Measure voltage ref  
 *
 ****************************************************************************/
event result_t Timer.fired() {
   uint8_t l_state;
        
   if (bLedOn){
    // call Leds.redOff();
     bLedOn = FALSE;
   }
   else{
     call Leds.redOn();
	 bLedOn = TRUE;
   }


   if (!bIOon){        //turn IO pins on/off
     SET_INT0();
	 SET_INT1();
	 SET_PW0();
	 SET_PW1();
	 SET_PWM1B();
	 SET_GPS_ENA();
	 bIOon = TRUE;
   }
   else{
     CLR_INT0();
	 CLR_INT1();
	 CLR_PW0();
	 CLR_PW1();
	 CLR_PWM1B();
	 CLR_GPS_ENA();
	 bIOon = FALSE;
  }
   	     
  atomic l_state = state;
  switch (l_state) { 
     case STATE0:
	     CLEAR_THERM_POWER();              //turn off thermistor power
         SET_BAT_MONITOR();                //turn on voltage ref power
		 atomic bGetVoltRef = TRUE;
		 TOSH_uwait(1000);                 //allow time to turn on
         call ADCBATT.getData();           //get sensor data;
		 atomic state = STATE1;
		 break;
     case STATE1:
	     CLEAR_BAT_MONITOR();              //turn off power to voltage ref     
         SET_THERM_POWER();                //turn on thermistor power
         atomic bGetVoltRef = FALSE;
	     TOSH_uwait(1000);
         call ADCBATT.getData();           //get sensor data;
	     break;
    
   }
 
   
   return SUCCESS;  
  }
/****************************************************************************
 * Battery Ref  or thermistor data ready 
 ****************************************************************************/
  async event result_t ADCBATT.dataReady(uint16_t data) {
  
    if (bGetVoltRef) {
	  msg_uart->data[VREF] = data & 0xff;      //voltage reference data
      msg_uart-> data[VREF+1] = data >> 8;
   	}
	else {
	   msg_uart->data[THERM] = data & 0xff;
	   msg_uart->data[THERM+1] = data >> 8;
	   call ADC2.getData();                    //get sensor data;
   }
   return SUCCESS;
  }
 /****************************************************************************
 * ADC data ready 
 * Read and get next channel.
 ****************************************************************************/ 
  async event result_t ADC2.dataReady(uint16_t data) {
       msg_uart->data[ADC2_D] = data & 0xff;
	   msg_uart->data[ADC2_D+1] = data >> 8;
	   call ADC3.getData();         //get sensor data;
       return SUCCESS;
   }
 /****************************************************************************
 * ADC data ready 
 * Read and get next channel.
 ****************************************************************************/ 
  async event result_t ADC3.dataReady(uint16_t data) {
       msg_uart->data[ADC3_D] = data & 0xff;
	   msg_uart->data[ADC3_D+1] = data >> 8;
	   call ADC4.getData();         //get sensor data;
       return SUCCESS;
   }
 /****************************************************************************
 * ADC data ready 
 * Read and get next channel.
 ****************************************************************************/ 
  async event result_t ADC4.dataReady(uint16_t data) {
       msg_uart->data[ADC4_D] = data & 0xff;
	   msg_uart->data[ADC4_D+1] = data >> 8;
	   call ADC5.getData();         //get sensor data;
       return SUCCESS;
   }
 /****************************************************************************
 * ADC data ready 
 * Read and get next channel.
 ****************************************************************************/ 
  async event result_t ADC5.dataReady(uint16_t data) {
       msg_uart->data[ADC5_D] = data & 0xff;
	   msg_uart->data[ADC5_D+1] = data >> 8;
	   call ADC6.getData();         //get sensor data;
       return SUCCESS;
   }
 /****************************************************************************
 * ADC data ready 
 * Read and get next channel.
 ****************************************************************************/ 
  async event result_t ADC6.dataReady(uint16_t data) {
       msg_uart->data[ADC6_D] = data & 0xff;
	   msg_uart->data[ADC6_D+1] = data >> 8;
	   call ADC7.getData();         //get sensor data;
       return SUCCESS;
   }
 /****************************************************************************
 * ADC data ready 
 * Read and get next channel.
 * Send data packet
 ****************************************************************************/ 
  async event result_t ADC7.dataReady(uint16_t data) {
       msg_uart->data[ADC7_D] = data & 0xff;
	   msg_uart->data[ADC7_D+1] = data >> 8;
       post send_uart_msg();
       atomic state = STATE0;            
       return SUCCESS;
  }
/****************************************************************************
 * Uart msg xmitted. 
 * Xmit same msg over radio
 ****************************************************************************/
  event result_t UARTSend.sendDone(TOS_MsgPtr msg, result_t success) {
   atomic msg_uart = msg;   
   post send_radio_msg();
	return SUCCESS;
  }
/****************************************************************************
 * Radio msg xmitted. 
 ****************************************************************************/
  event result_t RadioSend.sendDone(TOS_MsgPtr msg, result_t success) {
    msg_radio = msg;
    call Leds.redOff();
	return SUCCESS;
  }

/****************************************************************************
 * Radio msg rcvd. 
 * This app doesn't respond to any incoming radio msg
 * Just return
 ****************************************************************************/
  event TOS_MsgPtr RadioReceive.receive(TOS_MsgPtr data) {
     return data;
  }
/****************************************************************************
 * Uart msg rcvd. 
 * This app doesn't respond to any incoming uart msg
 * Just return
 ****************************************************************************/
  event TOS_MsgPtr UARTReceive.receive(TOS_MsgPtr data) {
    return data;
  }

}

