/*									tab:4
 *  IMPORTANT: READ BEFORE DOWNLOADING, COPYING, INSTALLING OR USING.  By
 *  downloading, copying, installing or using the software you agree to
 *  this license.  If you do not agree to this license, do not download,
 *  install, copy or use the software.
 *
 *  Intel Open Source License 
 *
 *  Copyright (c) 2002 Intel Corporation 
 *  All rights reserved. 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are
 *  met:
 * 
 *	Redistributions of source code must retain the above copyright
 *  notice, this list of conditions and the following disclaimer.
 *	Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *      Neither the name of the Intel Corporation nor the names of its
 *  contributors may be used to endorse or promote products derived from
 *  this software without specific prior written permission.
 *  
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *  PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE INTEL OR ITS
 *  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 *  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * 
 * $Id$
 */

/******************************************************************************
 *
 *****************************************************************************/
includes sensorboard;
module TestSensorM {
  provides {
    interface StdControl;
  }
  uses {
  
//UART communication
    interface StdControl as UARTControl;
    interface BareSendMsg as UARTSend;
    interface ReceiveMsg as UARTReceive;

//RF communication
    interface StdControl as RadioControl;
    interface BareSendMsg as RadioSend;
    interface ReceiveMsg as RadioReceive;
    
    interface ADC as ADCBATT;

//Temp
    interface StdControl as TempControl;
    interface ADC as Temperature;

//Light
    interface StdControl as PhotoControl;
    interface ADC as Light;
    
// Mic
    interface StdControl as MicControl;
    interface Mic;
    interface ADC as MicADC;

// Sounder
   interface StdControl as Sounder;


// Accel   
    interface StdControl as AccelControl;
    interface ADC as AccelX;
    interface ADC as AccelY;
    
// Mag
    interface StdControl as MagControl;
    interface ADC as MagX;
	interface ADC as MagY;

    
    interface ADCControl;   
    interface Timer;
    interface Leds;
  }
}

implementation {
	
  enum { START, BUSY, SOUND_DONE};

  enum { SENSOR_ID = 0,PACKET_ID, NODE_ID=2,RSVD, 
         VREF=4, THERM =6, PHOTO = 8, MIC = 10, ACCEL_X=12,ACCEL_Y=14,MAG_X=16, MAG_Y=18 };
  
  #define MSG_LEN  29 

   TOS_Msg msg_buf_uart, msg_buf_radio;
   TOS_MsgPtr msg_uart,msg_radio;
 
   char state;
   bool sound_state, sending_packet;

/****************************************************************************
 * Task to xmit radio message
 *
 ****************************************************************************/
   task void send_radio_msg(){
    msg_radio->data[SENSOR_ID] = SENSOR_BOARD_ID;
    msg_radio->data[PACKET_ID] = 1;    
    msg_radio->data[NODE_ID] = TOS_LOCAL_ADDRESS;
    msg_radio->data[RSVD] = 0;
    msg_radio->addr = TOS_BCAST_ADDR;
	msg_radio->type = 0;
	msg_radio->length = MSG_LEN;
	msg_radio->group = TOS_AM_GROUP;
    call RadioSend.send(msg_radio);
    return;
  }
/****************************************************************************
 * Task to uart as message
 *
 ****************************************************************************/
   task void send_uart_msg(){
   uint8_t i;
   atomic sending_packet=TRUE;

    msg_uart->data[SENSOR_ID] = SENSOR_BOARD_ID;
    msg_uart->data[PACKET_ID] = 1; 
    msg_uart->data[NODE_ID] = TOS_LOCAL_ADDRESS;
    msg_uart->data[RSVD] = 0;
    msg_uart->addr = TOS_UART_ADDR;
    msg_uart->type = 0;
    msg_uart->length = MSG_LEN;
    msg_uart->group = TOS_AM_GROUP;  
    for (i = 0; i <= MSG_LEN-1; i++) msg_radio->data[i] = msg_uart->data[i];
    call UARTSend.send(msg_uart);
    return;
  }

 /****************************************************************************
 * Initialize the component. Initialize ADCControl, Leds
 *
 ****************************************************************************/
  command result_t StdControl.init() {
  	
  	atomic {
    msg_uart = &msg_buf_uart;
    msg_radio = &msg_buf_radio;
    }
  	
  	MAKE_BAT_MONITOR_OUTPUT();  // enable voltage ref power pin as output
	MAKE_ADC_INPUT();           // enable ADC7 as input

// usart1 is also connected to external serial flash
// set usart1 lines to correct state
//  TOSH_MAKE_FLASH_SELECT_OUTPUT();
    TOSH_MAKE_FLASH_OUT_OUTPUT();             //tx output
    TOSH_MAKE_FLASH_CLK_OUTPUT();             //usart clk
//  TOSH_SET_FLASH_SELECT_PIN();
  	
  	call ADCControl.init();
    call Leds.init();
	call UARTControl.init();
    call RadioControl.init();
    call TempControl.init();
    call MicControl.init();
    call Mic.gainAdjust(64);  // Set the gain of the microphone.  (refer to Mic.ti)
    call Sounder.init();
    atomic {
  		sending_packet=FALSE;
  		state = START;
  		sound_state = TRUE;
  		}
    
    
    #ifdef MTS310
    call AccelControl.init();
    call MagControl.init();
    #endif
    
   	return SUCCESS;

  }
 /****************************************************************************
 * Start the component. Start the clock.
 *
 ****************************************************************************/
  command result_t StdControl.start(){
  	call Leds.redOn();
    call Leds.yellowOn();
    call Leds.greenOn();
  	
  	
  	call TempControl.start(); 
  	#ifdef MTS310
    call AccelControl.start();
    call MagControl.start();
    #endif
    call UARTControl.start();
    call RadioControl.start();
	call Timer.start(TIMER_REPEAT, 1000);
    return SUCCESS;	
  }
 /****************************************************************************
 * Stop the component.
 *
 ****************************************************************************/
  command result_t StdControl.stop() {
  	call TempControl.stop();  
  	call PhotoControl.stop(); 
  	#ifdef MTS310
    call AccelControl.stop();
    call MagControl.stop();
    #endif 
    call UARTControl.stop();
    call RadioControl.stop();
    return SUCCESS;    
  }
/****************************************************************************
 * Measure Temp, Light, Mic, toggle sounder  
 *
 ****************************************************************************/
event result_t Timer.fired() {
    char l_state;
   
    
            
	call Leds.redToggle();
    call Leds.yellowOff();
    call Leds.greenOn();
    atomic l_state = state;
    
    if (sending_packet || (l_state == BUSY)) return SUCCESS ;      //don't overrun buffers
	
    if ((l_state == START) ||( l_state == SOUND_DONE)){
    	atomic state = BUSY;
      	SET_BAT_MONITOR();                //turn on voltage ref power
		TOSH_uwait(100);                  //allow time to turn on
        call ADCBATT.getData();           //get sensor data;
           	
    }
   	return SUCCESS;
  }
  
  /****************************************************************************
 * Battery Ref  or thermistor data ready 
 ****************************************************************************/
  async event result_t ADCBATT.dataReady(uint16_t data) {
      msg_uart-> data[VREF] = data & 0x00ff;      //voltage reference data
      msg_uart-> data[VREF+1] = data >> 8;
      CLEAR_BAT_MONITOR();
      call Temperature.getData(); 
        
      return SUCCESS;
  }
  
    
/****************************************************************************
 * Temperature ADC data ready 
 * Read and get next channel.
 ****************************************************************************/ 
  async event result_t Temperature.dataReady(uint16_t data) {
       msg_uart->data[THERM] = data & 0xff;
	   msg_uart->data[THERM+1] = data >> 8;
       call TempControl.stop();  
       call PhotoControl.init();
       call PhotoControl.start(); 
	   call Light.getData(); 
       return SUCCESS;
  }

  
/****************************************************************************
 * Photocell ADC data ready 
 * Read and get next channel.
 ****************************************************************************/ 
  async event result_t Light.dataReady(uint16_t data) {
       msg_uart->data[PHOTO] = data & 0xff;
	   msg_uart->data[PHOTO+1] = data >> 8;
              
       call PhotoControl.stop(); 
 	   call TempControl.init();
       call TempControl.start(); 
 	   call MicADC.getData(); 
       
       return SUCCESS;
  }

/****************************************************************************
 * MicroPhone ADC data ready 
 * Read and toggle sounder.
 * send uart packet
 ****************************************************************************/                  
  async event result_t MicADC.dataReady(uint16_t data)
  {
  	  msg_uart->data[MIC] = data & 0xff;
	  msg_uart->data[MIC+1] = data >> 8;
              
      #ifndef MTS310
    
    if(!sending_packet)post send_uart_msg();
    call Leds.yellowOn();
      if(sound_state) call Sounder.start();
      else call Sounder.stop();
      atomic {
      	sound_state = !sound_state;
      	atomic state = SOUND_DONE;
    }
    #else
    call AccelX.getData();
    #endif
    return SUCCESS;
      
      
  } 
  
 
/****************************************************************************
 *  ADC data ready 
 * Read and toggle sounder.
 * send uart packet
 ****************************************************************************/                  
  async event result_t AccelX.dataReady(uint16_t data) {
		msg_uart->data[ACCEL_X]   = data & 0x00ff;
		msg_uart->data[ACCEL_X+1] = data >> 8;
        call AccelY.getData();

    	return SUCCESS;
  	
  	
}

/****************************************************************************
 *  ADC data ready 
 * Read and toggle sounder.
 * send uart packet
 ****************************************************************************/                  
  async event result_t AccelY.dataReady(uint16_t data) {
		msg_uart->data[ACCEL_Y]   = data & 0x00ff;
		msg_uart->data[ACCEL_Y+1] = data >> 8;
        call MagX.getData();
		//post send_uart_msg();
    	return SUCCESS;
  	
  	
}

  /**
   *  In response to the <code>MagX.dataReady</code> event, it stores the sample
   *  and issues command to sample the magnetometer's Y axis. (Magnetometer B pin)
   *  
   *  @return returns <code>SUCCESS</code>
   *
   **/
  async event result_t MagX.dataReady(uint16_t data){
    	msg_uart->data[MAG_X]   = data & 0x00ff;
		msg_uart->data[MAG_X+1] = data >> 8;
    	call  MagY.getData(); //get data for MagnetometerB
    	return SUCCESS;  
  }

  /**
   *  In response to the <code>MagY.dataReady</code> event, it stores the sample
   *  and issues a task to filter and process the stored magnetometer data.
   *
   *  It also has a schedule which starts sampling the Temperture and Accelormeter depending
   *  on the stepdown counter.
   * 
   *  @return returns <code>SUCCESS</code>
   **/
  async event result_t MagY.dataReady(uint16_t data){
    	msg_uart->data[MAG_Y]   = data & 0x00ff;
		msg_uart->data[MAG_Y+1] = data >> 8;
		//atomic sending_packet=TRUE;
    	post send_uart_msg();
        call Leds.yellowOn();
        if(sound_state) call Sounder.start();
        else call Sounder.stop();
        atomic {
      	sound_state = !sound_state;
      	atomic state = SOUND_DONE;
        }
        return SUCCESS;  
  }


  
  
/****************************************************************************
 * Uart msg xmitted. 
 * Xmit same msg over radio
 ****************************************************************************/
  event result_t UARTSend.sendDone(TOS_MsgPtr msg, result_t success) {
   atomic msg_uart = msg;   
   post send_radio_msg();
	return SUCCESS;
  }
/****************************************************************************
 * Radio msg xmitted. 
 ****************************************************************************/
  event result_t RadioSend.sendDone(TOS_MsgPtr msg, result_t success) {
    msg_radio = msg;
    call Leds.redOff();
    atomic sending_packet=FALSE;
	return SUCCESS;
  }

/****************************************************************************
 * Radio msg rcvd. 
 * This app doesn't respond to any incoming radio msg
 * Just return
 ****************************************************************************/
  event TOS_MsgPtr RadioReceive.receive(TOS_MsgPtr data) {
     return data;
  }
/****************************************************************************
 * Uart msg rcvd. 
 * This app doesn't respond to any incoming uart msg
 * Just return
 ****************************************************************************/
  event TOS_MsgPtr UARTReceive.receive(TOS_MsgPtr data) {
    return data;
  }

}

