/*									tab:4
 *  IMPORTANT: READ BEFORE DOWNLOADING, COPYING, INSTALLING OR USING.  By
 *  downloading, copying, installing or using the software you agree to
 *  this license.  If you do not agree to this license, do not download,
 *  install, copy or use the software.
 *
 *  Intel Open Source License 
 *
 *  Copyright (c) 2002 Intel Corporation 
 *  All rights reserved. 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are
 *  met:
 * 
 *	Redistributions of source code must retain the above copyright
 *  notice, this list of conditions and the following disclaimer.
 *	Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the distribution.
 *      Neither the name of the Intel Corporation nor the names of its
 *  contributors may be used to endorse or promote products derived from
 *  this software without specific prior written permission.
 *  
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *  PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE INTEL OR ITS
 *  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 *  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * 
 * $Id$
 */

/******************************************************************************
 *
 *    -Tests the MTS510 Mica2Dot Sensor Board
 *     Reads the light and accelerometer sensor readings
 *     Reads a sound sample
 *-----------------------------------------------------------------------------
 * Output results through mica2dot uart and radio. 
 * Use Xlisten.exe program to view data from either port:
 *  uart: mount mica2dot on mib510 with MTS510
 *        connect serial cable to PC
 *        run xlisten.exe at 19200 baud
 *  radio: run mica2dot with or without MTS510, 
 *         run mica2 with TOSBASE
 *         run xlisten.exe at 57600 baud
 *-----------------------------------------------------------------------------
 * Data packet structure  :
 *  msg->data[0] : sensor id, MTS510 = 0x02
 *  msg->data[1] : packet id
 *  msg->data[2] : node id
 *  msg->data[3] : reserved
 *  msg->data[4,5] : Light ADC data
 *  msg->data[6,7] : ACCEL - X-axis data
 *  msg->data[8,9] : ACCEL - Y-axis data
 *  msg->data[10,11] : Sound sample 0
 *  msg->data[12,13] : Sound sample 1
 *  msg->data[14,15] : Sound sample 2
 *  msg->data[16,17] : Sound sample 3
 *  msg->data[18,19] : Sound sample 4
 * 
 *------------------------------------------------------------------------------
 *
 *****************************************************************************/  

#define STATE_WAITING 0
#define STATE_LIGHT   1
#define STATE_ACCEL   2
#define STATE_SOUND   3

#define SOUNDSAMPLES  5

includes sensorboard;

module TestSensorMTS510M 
{
  provides interface StdControl;
  uses 
  {
    interface StdControl as UARTControl;
    interface BareSendMsg as UARTSend;
    interface ReceiveMsg as UARTReceive;

    interface Leds;
    interface Timer;

    interface StdControl as AccelControl;
    interface ADC as AccelX;
    interface ADC as AccelY;
    interface StdControl as MicControl;
    interface ADC as MicADC;
    interface Mic;
    interface ADC as PhotoADC;
    interface StdControl as PhotoControl;

    interface StdControl as RadioControl;
    interface BareSendMsg as RadioSend;
    interface ReceiveMsg as RadioReceive;
    

  }
}

implementation
{
enum { SENSOR_ID = 0, PACKET_ID, NODE_ID, RSVD, LIGHT = 4,
       ACCEL_X = 6,
       ACCEL_Y = 8,
       SOUND = 10
	};

#define MSG_LEN  29

  TOS_Msg msg_buf_uart, msg_buf_radio;
  TOS_MsgPtr msg_uart,msg_radio;

  bool sendPending;
  uint8_t samplecount;
  uint8_t state;

task void send_radio_msg() {

  
   	msg_radio->data[SENSOR_ID] = 0x02;
        msg_radio->data[PACKET_ID] = 0x01;
        msg_radio->data[NODE_ID] = TOS_LOCAL_ADDRESS;
        msg_radio->data[RSVD] = 0;

        msg_radio->addr = TOS_BCAST_ADDR;
        msg_radio->type = 0;
        msg_radio->length = MSG_LEN;
        msg_radio->group = TOS_AM_GROUP;

        call RadioSend.send(msg_radio);
        return;

}

task void send_uart_msg() {

uint8_t i;        

   	msg_uart->data[SENSOR_ID] = 0x02;
        msg_uart->data[PACKET_ID] = 0x01;
        msg_uart->data[NODE_ID] = TOS_LOCAL_ADDRESS;
        msg_uart->data[RSVD] = 0;

        msg_uart->addr = TOS_UART_ADDR;
        msg_uart->type = 0;
        msg_uart->length = MSG_LEN;
        msg_uart->group = TOS_AM_GROUP;

        for (i = 2; i <= MSG_LEN - 1; i++)
            msg_radio->data[i] = msg_uart->data[i];

        call UARTSend.send(msg_uart);
        return;

}



  /*************************************** 
     initialize lower components.
  ***************************************/
  command result_t StdControl.init() 
  {

    msg_uart = &msg_buf_uart;
    msg_radio = &msg_buf_radio;

    sendPending = TRUE;
    call RadioControl.init();
    call UARTControl.init();
    call Leds.init();
    sendPending = FALSE;
    call MicControl.init();
    call Mic.gainAdjust(64);
    call PhotoControl.init();
    call AccelControl.init();
    state = STATE_WAITING;
    samplecount = 0;

    call Leds.greenOff(); 
    call Leds.yellowOff(); 
    call Leds.redOff(); 

    call Leds.redOn();
    TOSH_uwait(1000);
    call Leds.redOff();

    return SUCCESS;
  }

  command result_t StdControl.start() 
  {
    call MicControl.start();
    call PhotoControl.start();
    call UARTControl.start();
    call RadioControl.start();
    call Timer.start(TIMER_REPEAT, 500);

    state = STATE_LIGHT;

    return SUCCESS;
  }

  command result_t StdControl.stop() 
  {
    call Timer.stop();
    call RadioControl.stop();
    call UARTControl.stop();
    call MicControl.stop();
    call PhotoControl.stop();

    return SUCCESS;
  }

/*********************************************
event handlers
*********************************************/

/***********************************************/  
  event result_t Timer.fired() 
  {
    if (state == STATE_LIGHT) {
      call PhotoADC.getData();
    }
    return SUCCESS;
  }


/*******************************************/
  async event result_t PhotoADC.dataReady(uint16_t data)
  {

	msg_uart->data[LIGHT] = data & 0x00ff;
	msg_uart->data[LIGHT+1] = data >> 8;

        call AccelX.getData();

    return SUCCESS;
  }  

/**********************************************/
  async event result_t AccelX.dataReady(uint16_t  data)
  {
	msg_uart->data[ACCEL_X]   = data & 0x00ff;
	msg_uart->data[ACCEL_X+1] = data >> 8;
        call AccelY.getData();

    return SUCCESS;
  }

/**************************************************/
  async event result_t AccelY.dataReady(uint16_t  data)
  {

	msg_uart->data[ACCEL_Y] = data & 0x00ff;
	msg_uart->data[ACCEL_Y+1] = data >> 8;

        call MicADC.getData();

    return SUCCESS;
  }


/***************************************************/    
async event result_t MicADC.dataReady(uint16_t data)
{

    atomic {
       msg_uart->data[SOUND + samplecount * 2] = data & 0x00FF;
       msg_uart->data[SOUND + (samplecount * 2) + 1] = data >> 8;

       samplecount++;
       if (samplecount == SOUNDSAMPLES) {
           samplecount = 0;
           post send_uart_msg();
       } else { 
           call MicADC.getData();
       }
    }

    return SUCCESS;
}

/***************************************************/    
event result_t UARTSend.sendDone(TOS_MsgPtr msg, result_t success) {
      msg_uart = msg;
      post send_radio_msg();
      return SUCCESS;
}

event result_t RadioSend.sendDone(TOS_MsgPtr msg, result_t success) {
      msg_radio = msg;
      return SUCCESS;
}

/***************************************************/    
event TOS_MsgPtr UARTReceive.receive(TOS_MsgPtr data) {
    return data;
}

event TOS_MsgPtr RadioReceive.receive(TOS_MsgPtr data) {
    return data;
}

} 
