/**
 * Handles conversion to engineering units of mda300 packets.
 *
 * @file      mda300.c
 * @author    Martin Turon
 * @version   2004/3/23    mturon      Initial version
 *
 * Copyright (c) 2004 Crossbow Technology, Inc.   All rights reserved.
 * 
 * $Id$
 */

#include <math.h>
#include "../xsensors.h"

/** MDA300 XSensor packet 1 -- contains single analog adc channels */
typedef struct {
    uint16_t adc0;
    uint16_t adc1;
    uint16_t adc2;
    uint16_t adc3;
    uint16_t adc4;
    uint16_t adc5;
    uint16_t adc6;
} XSensorMDA300Data1;

/** MDA300 XSensor packet 2 -- contains precision analog adc channels. */
typedef struct {
    uint16_t adc7;
    uint16_t adc8;
    uint16_t adc9;
    uint16_t adc10;
    uint16_t adc11;
    uint16_t adc12;
    uint16_t adc13;
} XSensorMDA300Data2;

/** MDA300 XSensor packet 3 -- contains digital channels. */
typedef struct {
    uint16_t digi0;
    uint16_t digi1;
    uint16_t digi2;
    uint16_t digi3;
    uint16_t digi4;
    uint16_t digi5;
} XSensorMDA300Data3;

/** MDA300 XSensor packet 4 -- contains misc other sensor data. */
typedef struct {
    uint16_t battery;
    uint16_t humidity;
    uint16_t temperature;
    uint16_t counter;
} XSensorMDA300Data4;

uint32_t mda300_convert_adc_single    (uint16_t adc_prec);
int32_t  mda300_convert_adc_precision (uint16_t adc_prec); 
float    mda300_convert_temperature   (XSensorMDA300Data4 *data); 
uint16_t mda300_convert_humidity      (XSensorMDA300Data4 *data); 
uint16_t mda300_convert_battery       (XSensorMDA300Data4 *data);

/** 
 * MDA300 Specific outputs of raw readings within a XSensor packet.
 *
 * @author    Martin Turon
 *
 * @version   2004/3/23       mturon      Initial version
 */
void mda300_print_raw(XbowSensorboardPacket *packet) 
{
    switch (packet->packet_id) {
        case 1: {
            XSensorMDA300Data1 *data = (XSensorMDA300Data1 *)packet->data;
            printf("mda300 id=%02x a0=%04x a1=%04x a2=%04x a3=%04x "
                   "a4=%04x a5=%04x a6=%04x\n",
                   packet->node_id, data->adc0, data->adc1, 
                   data->adc2, data->adc3, data->adc4, 
                   data->adc5, data->adc6);
            break;
        }

        case 2: {
            XSensorMDA300Data2 *data = (XSensorMDA300Data2 *)packet->data;
            printf("mda300 id=%02x a7=%04x a8=%04x a9=%04x a10=%04x "
                   "a11=%04x a12=%04x a13=%04x\n",
                   packet->node_id, data->adc7, data->adc8, 
                   data->adc9, data->adc10, data->adc11, 
                   data->adc12, data->adc13);
            break;
        }

        case 3: {
            XSensorMDA300Data3 *data = (XSensorMDA300Data3 *)packet->data;
            printf("mda300 id=%02x d1=%04x d2=%04x d3=%04x d4=%04x d5=%04x\n",
                   packet->node_id, data->digi0, data->digi1, 
                   data->digi2, data->digi3, data->digi4, data->digi5);
            break;
        }

        case 4: {
            XSensorMDA300Data4 *data = (XSensorMDA300Data4 *)packet->data;
            printf("mda300 id=%02x bat=%04x hum=%04x temp=%04x cntr=%04x\n",
                   packet->node_id, data->battery, data->humidity, 
                   data->temperature, data->counter);
            break;
        }

        default:
            printf("mda300 error: unknown packet_id (%i)\n",packet->packet_id);
    }
}

/** MDA300 specific display of converted readings for packet 1 */
void mda300_print_cooked_1(XbowSensorboardPacket *packet)
{
    printf("MDA300 [sensor data converted to engineering units]:\n"
           "   health:     node id=%i packet=%i\n"
           "   adc chan 0: voltage=%i mV\n"
           "   adc chan 1: voltage=%i mV\n"
           "   adc chan 2: voltage=%i mV\n"
           "   adc chan 3: voltage=%i mV\n" 
           "   adc chan 4: voltage=%i mV\n" 
           "   adc chan 5: voltage=%i mV\n" 
           "   adc chan 6: voltage=%i mV\n\n",
           packet->node_id, packet->packet_id,
           mda300_convert_adc_single(packet->data[0]),
           mda300_convert_adc_single(packet->data[1]),
           mda300_convert_adc_single(packet->data[2]),
           mda300_convert_adc_single(packet->data[3]),
           mda300_convert_adc_single(packet->data[4]),
           mda300_convert_adc_single(packet->data[5]),
           mda300_convert_adc_single(packet->data[6]));
}

/** MDA300 specific display of converted readings  for packet 2 */
void mda300_print_cooked_2(XbowSensorboardPacket *packet)
{
    printf("MDA300 [sensor data converted to engineering units]:\n"
           "   health:      node id=%i packet=%i\n"
           "   adc chan 7:  voltage=%i uV\n"
           "   adc chan 8:  voltage=%i uV\n"
           "   adc chan 9:  voltage=%i uV\n"
           "   adc chan 10: voltage=%i uV\n" 
           "   adc chan 11: voltage=%i mV\n" 
           "   adc chan 12: voltage=%i mV\n" 
           "   adc chan 13: voltage=%i mV\n\n",
           packet->node_id, packet->packet_id,
           mda300_convert_adc_precision(packet->data[0]),
           mda300_convert_adc_precision(packet->data[1]),
           mda300_convert_adc_precision(packet->data[2]),
           mda300_convert_adc_precision(packet->data[3]),
           mda300_convert_adc_single(packet->data[4]),
           mda300_convert_adc_single(packet->data[5]),
           mda300_convert_adc_single(packet->data[6]));
}

/** MDA300 specific display of converted readings for packet 3 */
void mda300_print_cooked_3(XbowSensorboardPacket *packet)
{
    printf("MDA300 [sensor data converted to engineering units]:\n"
           "   health:     node id=%i packet=%i\n\n",
           packet->node_id, packet->packet_id);
}

/** MDA300 specific display of converted readings for packet 4 */
void mda300_print_cooked_4(XbowSensorboardPacket *packet)
{
	XSensorMDA300Data4 *data = (XSensorMDA300Data4 *)packet->data;
    printf("MDA300 [sensor data converted to engineering units]:\n"
           "   health:     node id=%i packet=%i\n"
           "   battery voltage:   =%i mV  \n"
           "   temperature:       =%0.2f C \n"
           "   humidity:          =%i %% \n\n",
           packet->node_id, packet->packet_id, 
	           mda300_convert_battery(data),
	           mda300_convert_temperature(data), 
		   mda300_convert_humidity(data));
}

/** MDA300 specific display of converted readings from an XSensor packet. */
void mda300_print_cooked(XbowSensorboardPacket *packet) 
{
    switch (packet->packet_id) {
        case 1:
            mda300_print_cooked_1(packet);
            break;

        case 2:
            mda300_print_cooked_2(packet);
            break;

        case 3:
            mda300_print_cooked_3(packet);
            break;

        case 4:
            mda300_print_cooked_4(packet);
            break;
        
        default:
            printf("MDA300 Error: unknown packet id (%i)\n\n", packet->packet_id);
    }
}

/** 
 * Converts raw MDA300 temperature to degrees Celcius.
 *
 * @author    Martin Turon
 *
 * @version   2004/4/2       mturon      Initial version
 *
 */
float mda300_convert_temperature(XSensorMDA300Data4 *data) 
{
	uint16_t temp = data->temperature;
	float t = (((float)(temp) )*0.98-3840)/100;
	if(t > 100 ) temp=100;      //centigrade limits    
	if(t < -15 ) temp=-15;      //centigrade limits    
    return   t;
}

/** 
 * Converts raw MDA300 ADC reading to % humidity. 
 * 
 * @author    Martin Turon
 *
 * @version   2004/4/2       mturon      Initial revision
 *
 */
uint16_t mda300_convert_humidity(XSensorMDA300Data4 *data) 
{
	uint16_t hum = data->humidity;

	float t = mda300_convert_temperature(data);
	if (data->temperature == 0) t=25;       // use good defaults

	float h;                                // temp humidity value.
    h= 0.0405 * (float) (hum) - 4 - (float)(hum) * (float)(hum)*0.0000028;
    h= (t-25) * (0.01 + 0.00128 * hum) + h;
    hum= (uint16_t) h;
    if (hum > 100) hum=100;
    return hum;
}

/** 
 * Computes the voltage of an adc channel using the reference voltage. 
 * Final formula is designed to minimize fixed point bit shifting 
 * round off errors.
 *
 * Convert 12 bit data to mV:  
 *     Dynamic range is 0 - 2.5V
 *     voltage = (adc_data * 2500mV) / 4096 
 *             = (adc_data * 625mV)  / 1024
 *
 * @author    Martin Turon
 *
 * @version   2004/3/24       mturon      Initial revision
 *
 */
uint32_t mda300_convert_adc_single(uint16_t adc_sing) 
{
    uint32_t analog_mV = (625 * (uint32_t)adc_sing) / 1024;
    return   analog_mV;
}

/** 
 * Computes the voltage of an adc channel using the reference voltage. 
 * Final formula is designed to minimize fixed point bit shifting 
 * round off errors.
 *
 * Convert 12 bit data to uV:
 *     Dynamic range is +/- 12.5mV
 *     voltage = 12500 * (adc_data/2048 -1) 
 *             = (5*625*data/512) - 12500 
 *             = 5 * ((625*data/512) - 2500)
 *
 *
 * @author    Martin Turon
 *
 * @version   2004/3/24       mturon      Initial revision
 *
 */
int32_t mda300_convert_adc_precision(uint16_t adc_prec) 
{
    int32_t analog_uV = 5 * (((625 * (uint32_t)adc_prec)/ 512) - 2500);
    return  analog_uV;
}

/** 
 * Converts mica2 battery reading from raw ADC data to engineering units.
 *
 * @author    Martin Turon, Jaidev Prabhu
 *
 * We get the reading from MDA300 in 10s of mVolts in ADC counts
 *
 * @version   2004/4/6       jdprabhu      Changed for MDA300
 *
 */
uint16_t mda300_convert_battery(XSensorMDA300Data4 *data) 
{
    float    x     = (float)data->battery;
    uint16_t vdata = (uint16_t) (x * 10);  
    return vdata;
}


