/**
 * Handles conversion to engineering units of mda500 packets.
 *
 * @file      mda500.c
 * @author    Martin Turon
 * @version   2004/3/10    mturon      Initial version
 *
 * Refer to:
 *   -Panasonic ERT-J1VR103J thermistor data sheet
 *   - or Xbow MTS300CA sensor board (i.e. micasb) (uses same thermistor)
 *   - or Xbow MTS/MDA Sensor and DataAcquistion Manual  
 *
 * Copyright (c) 2004 Crossbow Technology, Inc.   All rights reserved.
 * 
 * $Id$
 */

#include <math.h>
#include "../xsensors.h"

/** MDA500 XSensor packet 1 -- contains battery, thermistor, and adc2-7. */
typedef struct {
    uint16_t battery;
    uint16_t thermistor;
    uint16_t adc2;
    uint16_t adc3;
    uint16_t adc4;
    uint16_t adc5;
    uint16_t adc6;
    uint16_t adc7;
} XSensorMDA500Data;

uint16_t mda500_convert_battery (XbowSensorboardPacket *packet);
uint16_t mda500_convert_thermistor_resistance (XbowSensorboardPacket *packet);
float    mda500_convert_thermistor_temperature(XbowSensorboardPacket *packet);
uint16_t mda500_convert_adc(XbowSensorboardPacket *packet, uint16_t index); 

/** MDA500 Specific outputs of raw readings within an XBowSensorboardPacket */
void mda500_print_raw(XbowSensorboardPacket *packet) 
{
    XSensorMDA500Data *data = (XSensorMDA500Data *)packet->data;
    printf("mda500 id=%02x bat=%04x thrm=%04x a2=%04x a3=%04x "
           "a4=%04x a5=%04x a6=%04x a7=%04x\n",
           packet->node_id, data->battery, data->thermistor, 
           data->adc2, data->adc3, data->adc4, 
           data->adc5, data->adc6, data->adc7);
}

/** MDA500 specific display of converted readings from XBowSensorboardPacket */
void mda500_print_cooked(XbowSensorboardPacket *packet) 
{
    printf("MDA500 [sensor data converted to engineering units]:\n"
           "   health:     node id=%i\n"
           "   battery:    volts=%i mv\n"
           "   thermistor: resistance=%i ohms, tempurature=%0.2f C\n" 
           "   adc chan 2: voltage=%i mv\n"
           "   adc chan 3: voltage=%i mv\n" 
           "   adc chan 4: voltage=%i mv\n" 
           "   adc chan 5: voltage=%i mv\n" 
           "   adc chan 6: voltage=%i mv\n" 
           "   adc chan 7: voltage=%i mv\n", 
           packet->node_id,
           mda500_convert_battery(packet),
           mda500_convert_thermistor_resistance(packet),
           mda500_convert_thermistor_temperature(packet),
           mda500_convert_adc(packet, 2),
           mda500_convert_adc(packet, 3),
           mda500_convert_adc(packet, 4),
           mda500_convert_adc(packet, 5),
           mda500_convert_adc(packet, 6),
           mda500_convert_adc(packet, 7));
    printf("\n");
}

/** 
 * Converts battery reading from raw ADC data to engineering units.
 *
 * @author    Martin Turon, Alan Broad
 *
 * To compute the battery voltage after measuring the voltage ref:
 *   BV = RV*ADC_FS/data
 *   where:
 *   BV = Battery Voltage
 *   ADC_FS = 1023
 *   RV = Voltage Reference (0.6 volts)
 *   data = data from the adc measurement of channel 1
 *   BV (volts) = 614.4/data
 *   BV (mv) = 614400/data 
 *
 * Note:
 *   The thermistor resistance to temperature conversion is highly non-linear.
 *
 * @return    Battery voltage as uint16 in millivolts (mV)
 *
 * @version   2004/3/11       mturon      Initial revision
 *
 */
uint16_t mda500_convert_battery(XbowSensorboardPacket *packet) 
{
    XSensorMDA500Data *data = (XSensorMDA500Data *)packet->data;
    float    x     = (float)data->battery;
    uint16_t vdata = (uint16_t) (614400 / x);  /*613800*/
    return vdata;
}

/** 
 * Converts thermistor reading from raw ADC data to engineering units.
 *
 * @author    Martin Turon, Alan Broad
 *
 * To compute the thermistor resistance after measuring the thermistor voltage:
 * - Thermistor is a temperature variable resistor
 * - There is a 10K resistor in series with the thermistor resistor.
 * - Compute expected adc output from voltage on thermistor as: 
 *       ADC= 1023*Rthr/(R1+Rthr)
 *       where  R1 = 10K
 *              Rthr = unknown thermistor resistance
 *       Rthr = R1*ADC/(ADC_FS-ADC)
 *       where  ADC_FS = 1023
 *
 * Note:
 *   The thermistor resistance to temperature conversion is highly non-linear.
 *
 * @return    Thermistor resistance as a uint16 in unit (Ohms)
 *
 * @version   2004/3/11       mturon      Initial revision
 *
 */
uint16_t mda500_convert_thermistor_resistance(XbowSensorboardPacket *packet) 
{
    XSensorMDA500Data *data = (XSensorMDA500Data *)packet->data;
    float    x     = (float)data->thermistor;
    uint16_t vdata = 10000*x / (1023-x);
    return vdata;
}

/** 
 * Converts thermistor reading from raw ADC data to engineering units.
 *
 * @author    Martin Turon
 *
 * @return    Temperature reading from thermistor as a float in degrees Celcius
 *
 * @version   2004/3/22       mturon      Initial revision
 *
 */
float mda500_convert_thermistor_temperature(XbowSensorboardPacket *packet) 
{
    XSensorMDA500Data *data = (XSensorMDA500Data *)packet->data;

    float temperature, a, b, c, Rt;
    a  = 0.001307050;
    b  = 0.000214381;
    c  = 0.000000093;
    Rt = mda500_convert_thermistor_resistance(packet);

    temperature = 1 / (a + b * log(Rt) + c * pow(log(Rt),3));
    temperature -= 273.15;   // Convert from Kelvin to Celcius

    //printf("debug: a=%f b=%f c=%f Rt=%f temp=%f\n",a,b,c,Rt,temperature);

    return temperature;
}

/** 
 * Computes the voltage of an adc channel using the reference voltage. 
 *
 * @author    Martin Turon
 *
 * @return    Voltage of ADC channel as an unsigned integer in mV
 *
 * @version   2004/3/22       mturon      Initial revision
 *
 */
uint16_t mda500_convert_adc(XbowSensorboardPacket *packet, uint16_t index) 
{
    XSensorMDA500Data *data = (XSensorMDA500Data *)packet->data;
    float    Vbat = mda500_convert_battery(packet);
    uint16_t Vadc = (uint16_t) (packet->data[index] * Vbat / 1023);
    return (uint16_t)Vadc;
}

/*==========================================================================*/


/** MDA400 Specific outputs of raw readings within an XBowSensorboardPacket */
void mda400_print_raw(XbowSensorboardPacket *packet) 
{
    XSensorMDA500Data *data = (XSensorMDA500Data *)packet->data;
    printf("mda400 id=%02x bat=%04x thrm=%04x a2=%04x a3=%04x "
           "a4=%04x a5=%04x a6=%04x a7=%04x\n",
           packet->node_id, data->battery, data->thermistor, 
           data->adc2, data->adc3, data->adc4, 
           data->adc5, data->adc6, data->adc7);
}

/** MDA400 specific display of converted readings from XBowSensorboardPacket */
void mda400_print_cooked(XbowSensorboardPacket *packet) 
{
	printf("MDA400 [sensor data converted to engineering units]:\n"
		   "   health:     node id=%i\n"
		   "   battery:    volts=%i mv\n"
		   "   thermistor: resistance=%i ohms\n", 
                   packet->node_id,
		   mda500_convert_battery(packet),
		   mda500_convert_thermistor_resistance(packet));
	printf("\n");
}

