/**
 * Listens to the serial port, and outputs sensor data in human readable form.
 *
 * @file      xlisten.c
 * @author    Martin Turon
 * @version   2004/3/10    mturon      Initial version
 *
 * Copyright (c) 2004 Crossbow Technology, Inc.   All rights reserved.
 *
 * $Id$
 */

#include "xsensors.h"

const char *g_version = 
    "$Id$";

/** A structure to store parsed parameter flags. */
typedef union {
    unsigned flat;
    
    struct {
        // output display options
        unsigned  display_raw    : 1;  //!< raw TOS packets
        unsigned  display_parsed : 1;  //!< pull out sensor readings
        unsigned  display_cooked : 1;  //!< convert to engineering units
        unsigned  export_parsed  : 1;  //!< output comma delimited fields
        unsigned  display_rsvd   : 12; //!< pad first word for output options
        
        // modes of operation
        unsigned  display_help   : 1;
        unsigned  display_baud   : 1;  //!< baud was set by user
        unsigned  mode_quiet     : 1;  //!< suppress headers
        unsigned  mode_wireless  : 1;  //!< use wireless network headers
        unsigned  mode_debug     : 1;  //!< debug serial port
    } bits; 
    
    struct {
        unsigned short output;         //!< must have at least one output option
        unsigned short mode;
    } options;
} s_params;

/** A variable to store parsed parameter flags. */
s_params  g_params;


/**
 * Extracts command line options and sets flags internally.
 * 
 * @param     argc            Argument count
 * @param     argv            Argument vector
 *
 * @author    Martin Turon
 *
 * @version   2004/3/10       mturon      Intial version
 * @n         2004/3/12       mturon      Added -b,-s,-q,-x
 */
void parse_args(int argc, char **argv) 
{
    unsigned baudrate;
    g_params.flat = 0;   /* default to no params set */

    while (argc) {
        if ((argv[argc]) && (*argv[argc] == '-')) {
            switch(argv[argc][1]) {
                case '?':
                    g_params.bits.display_help = 1;
                    break;

                case 'q':
                    xserial_set_verbose(0);
                    g_params.bits.mode_quiet = 1;
                    break;

                case 'p':
                    g_params.bits.display_parsed = 1;
                    break;

                case 'r':
                    g_params.bits.display_raw = 1;
                    break;

                case 'c':
                    g_params.bits.display_cooked = 1;
                    break;

                case 'x':
                    g_params.bits.export_parsed = 1;
                    break;

                case 'w':
                    g_params.bits.mode_wireless = 1;
                    xpacket_set_start(XPACKET_DATASTART_WIRELESS);
                    break;

                case 'b':
                    if (argv[argc][2] == '=') {
                        baudrate = xserial_set_baud(argv[argc]+3);
                        g_params.bits.display_baud = 1;
                    }
                    break;

                case 's':
                    if (argv[argc][2] == '=') {
                        xserial_set_device(argv[argc]+3);
                    }
                    break;
					
                case 'd':
                    g_params.bits.mode_debug = 1;
                    break;
            }
        }
        argc--;
    }

    if (!g_params.bits.mode_quiet) {
        // Summarize parameter settings
        printf("xlisten Ver:%s\n", g_version);
        printf("Using params: ");
        if (g_params.bits.display_help)   printf("[help] ");
        if (g_params.bits.display_baud)   printf("[baud=0x%04x] ", baudrate);
        if (g_params.bits.display_raw)    printf("[raw] ");
        if (g_params.bits.display_parsed) printf("[parsed] ");
        if (g_params.bits.display_cooked) printf("[cooked] ");
        if (g_params.bits.export_parsed)  printf("[export] ");
        if (g_params.bits.mode_wireless)  printf("[wireless] ");
        if (g_params.bits.mode_debug) {
            printf("[debug - serial dump!] \n");
            xserial_port_dump();
        }
        printf("\n");
    }

    if (g_params.bits.display_help) {
        printf(
            "\nUsage: xlisten <-?|r|p|x|c|d|q> <-b=baud> <-s=device>"
            "\n   -? = display help [help]"
            "\n   -r = raw display of tos packets [raw]"
            "\n   -p = parse packet into raw sensor readings [parsed]"
            "\n   -x = export readings in csv spreadsheet format [export]"
            "\n   -c = convert data to engineering units [cooked]"
            "\n   -d = debug serial port by dumping bytes [debug]"
            "\n   -b = set the baudrate [baud=#|mica2|mica2dot]"
            "\n   -s = set serial port device [device=com1]"
            "\n   -w = read wireless network packets from TOSBase"
            "\n   -q = quiet mode (suppress headers)"
            "\n"
        );
        exit(0);
    }

    /* Default to displaying packets as raw, parsed, and cooked. */
    if (g_params.options.output == 0) {
	g_params.bits.display_raw = 1;
	g_params.bits.display_parsed = 1;
	g_params.bits.display_cooked = 1;
    }
}

/**
 * The main entry point for the sensor listener console application.
 * 
 * @param     argc            Argument count
 * @param     argv            Argument vector
 *
 * @author    Martin Turon
 * @version   2004/3/10       mturon      Intial version
 */
int main(int argc, char **argv) 
{
    parse_args(argc, argv); 

    char c;
    int serline, i, partial = 0;
    unsigned char buffer[64];

    serline = xserial_port_open();
    while (1) {
        if (xserial_port_read_packet(serline, buffer, XPACKET_SIZE))
	    continue;     // ignore patial packets and packetizer frame end

        if (g_params.bits.display_raw)    xpacket_print_raw(buffer);
	
	if (g_params.bits.mode_wireless)  xpacket_depacketize(buffer);

        if (g_params.bits.display_parsed) xpacket_print_parsed(buffer);

        if (g_params.bits.export_parsed)  xpacket_export_parsed(buffer);

        if (g_params.bits.display_cooked) xpacket_print_cooked(buffer);
    }
}


//####################### User Manual Follows ##############################

/** 
@mainpage XListen Documentation

@section version Version 
$Id$

@section usage Usage 
Usage: xlisten <-?|r|p|x|c|d|q> <-b=baud> <-s=device>
@n
@n      -? = display help [help]
@n      -r = raw display of tos packets [raw]
@n      -p = parse packet into raw sensor readings [parsed]
@n      -x = export readings in csv spreadsheet format [export]
@n      -c = convert data to engineering units [cooked]
@n      -d = debug serial port by dumping bytes [debug]
@n      -b = set the baudrate [baud=#|mica2|mica2dot]
@n      -s = set serial port device [device=com1]
@n      -w = read wireless network packets from TOSBase
@n      -q = quiet mode (suppress headers)
@n

@section params Parameters

@subsection help -? [help]

XListen has many modes of operation that can be controlled by passing command line parameters.  The current list of these command line options and a brief usage explanation is always available by passing the -? flag.
@n
@n A detail explanation of each command line option as of version 1.7 follows.

@subsection baud -b=baudrate [baud]
     This flag allows the user to set the baud rate of the serial line connection.  The default baud rate is 57600 bits per second which is compatible with the Mica2.  The desired baudrate must be passed as a  number directly after the equals sign with no spaces inbetween, i.e. -b=19200.  Optionally, a product name can be passed in lieu of an actual number and the proper baud will be set, i.e. -b=mica2dot.  Valid product names are:
	mica2           	(57600 baud)
	mica2dot	(19200 baud)


@subsection serial -s=port [serial]
     This flag gives the user the ability to specify which COM port or device xlisten should use.  The default port is /dev/ttyS0 or the UNIX equivalent to COM1.  The given port must be passed directly after the equals sign with no spaces, i.e. -s=com3.  

@subsection raw	-r [raw]
     Raw mode displays the actual TOS packets as a sequence of bytes as seen coming over the serial line.  Sample output follows:

@n $ xlisten -r
@n xlisten Ver: Id: xlisten.c,v 1.7 2004/03/23 00:52:28 mturon Exp
@n Using params: [raw]
@n /dev/ttyS0 input stream opened
@n 7e7e000033000000c8035f61d383036100000000e4510d610000000080070000d4b5f577
@n 7e00007d1d8101060029091e09ef082209e7080b09b40800000000000000000000000100
@n 7e00007d1d81020600f007de07da07d507c3064706540500000000000000000000000100

@subsection parsed	-p [parsed]
     Parsed mode attempts to interpret the results of the incoming TOS packets and display information accordingly.  The first stage of the parsing is to look for a valid sensorboard_id field, and display the part number.  The node_id of the packet sender is also pulled out and displayed.  Finally, raw sensor readings are extracted and displayed with some designation as to their meaning:

@n $ xlisten -p -b=mica2dot
@n xlisten Ver: Id: xlisten.c,v 1.7 2004/03/23 00:52:28 mturon Exp
@n Using params: [baud=0x000e] [parsed]
@n /dev/ttyS0 input stream opened
@n mda500 id=06 bat=00c1 thrm=0203 a2=019c a3=0149 a4=011d a5=012b a6=011b a7=0147
@n mda500 id=06 bat=00c2 thrm=0203 a2=019d a3=014d a4=011e a5=0131 a6=011b a7=0140
@n mda500 id=06 bat=00c2 thrm=0204 a2=0199 a3=014c a4=0125 a5=012a a6=011f a7=0147
@n mda500 id=06 bat=00c2 thrm=0204 a2=0198 a3=0148 a4=0122 a5=0131 a6=012d a7=0143
@n mda500 id=06 bat=00c2 thrm=0203 a2=019e a3=014e a4=0124 a5=012b a6=011c a7=0143
@n mda500 id=06 bat=00c2 thrm=0204 a2=019d a3=014c a4=011f a5=0135 a6=0133 a7=011d
@n mda500 id=06 bat=00c2 thrm=0205 a2=019a a3=014c a4=011e a5=0131 a6=012d a7=011c

@subsection cooked	-c [cooked]
     Cooked mode actually converts the raw sensor readings within a given packet into engineering units.  Sample output follows:

@n $ xlisten -c -b=mica2dot
@n xlisten Ver: Id: xlisten.c,v 1.7 2004/03/23 00:52:28 mturon Exp
@n Using params: [baud=0x000e] [cooked]
@n /dev/ttyS0 input stream opened
@n MDA500 [sensor data converted to engineering units]:
@n    health:     node id=6
@n    battery:    volts=3163 mv
@n    thermistor: resistance=10177 ohms, tempurature=24.61 C
@n    adc chan 2: voltage=1258 mv
@n    adc chan 3: voltage=1001 mv
@n    adc chan 4: voltage=893 mv
@n    adc chan 5: voltage=939 mv
@n    adc chan 6: voltage=875 mv
@n    adc chan 7: voltage=850 mv

@subsection quiet	-q [quiet]
     This flag suppresses the standard xlisten header which displays the version string and parameter selections. 

@subsection export	-x [export]
    Export mode displays raw adc values as comma delimited text for use in spreadsheet and data manipulation programs.  The user can pipe the output of xlisten in export mode to a file and load that file into Microsoft Excel to build charts of the information.  Sample output follows:

@n $ xlisten -b=mica2dot -q -x
@n 51200,24323,54113,899,97,0,58368,3409
@n 6,193,518,409,328,283,296,298
@n 6,194,517,410,330,292,310,300
@n 6,194,518,409,329,286,309,288
@n 6,194,517,411,331,287,297,300
@n 6,194,516,413,335,288,301,287

@subsection wireless	-w [wireless]
     Passing the wireless flag tells xlisten to use a different offset when parsing packets that are being forwarded by TOSBase.  This flag is required to see parsed and cooked values from motes over a wireless network.

@subsection debug	-d [debug]
     This flag puts xlisten in a mode so that it behaves exactly like the TinyOS raw listen tool (tinyos-1.x/tools/src/raw_listen.c.)  All other command line options except -b [baud] and -s[serial] will be ignored.  This mode is mainly used for compatibility and debugging serial port issues.  Individual bytes will be displayed as soon as they are read from the serial port with no post-processing.  In most cases -r [raw] is equivalent and preferred to using debug mode.

@subsection display	Display Options
     The -r, -p, and -c flags are considered display options.  These can be passed in various combinations to display multiple views of the same packet at once.  The default display mode when xlisten is invoked with no arguments is -r.  What follows is sample output for all three display options turned on at once:

@n $ xlisten -b=mica2dot -r -p -c
@n xlisten Ver: Id: xlisten.c,v 1.7 2004/03/23 00:52:28 mturon Exp
@n Using params: [baud=0x000e] [raw] [parsed] [cooked]
@n /dev/ttyS0 input stream opened
@n 7e7e000033000000c8035f61d383036100000000e4510d610000000080070000d4b5f577
@n 7e00007d1d01010600c200050293014401210135012f0122010000000000000000000100
@n mda500 id=06 bat=00c2 thrm=0205 a2=0193 a3=0144 a4=0121 a5=0135 a6=012f a7=0122
@n MDA500 [sensor data converted to engineering units]:
@n    health:     node id=6
@n    battery:    volts=3163 mv
@n    thermistor: resistance=10217 ohms, tempurature=24.53 C
@n    adc chan 2: voltage=1246 mv
@n    adc chan 3: voltage=1001 mv
@n    adc chan 4: voltage=893 mv
@n    adc chan 5: voltage=955 mv
@n    adc chan 6: voltage=936 mv
@n    adc chan 7: voltage=896 mv

@section building Build Process
     The source code for the xlisten tool is located at: /opt/tinyos-1.x/contrib/xbow/tools/src/xlisten.  
@n@n
    To build the tool, change to the xlisten source directory and run `make`. 
@n@n
    To get the latest version of the source, change to the xlisten source directory and run `cvs update`.  


@section setup Setup

    XListen is a command line tool that can be run from a cygwin shell by simply typing `xlisten`.  The executable needs to be in your working path to use it.  A simple way to add xlisten to your working path is to create a soft link to it by running the following command: 
@n$ ln -s /opt/tinyos-1.x/contrib/xbow/tools/src/xlisten /usr/local/bin/xlisten
@n@n
  You can use xlisten to read sensor data from either one mote over a serial link, or a wireless network of motes.  In both configurations, you need to have a MIB510 board connected via a serial cable to your PC.
@n@n
    For a single mote configuration, the mote must be programmed with a XSensorMXX### application and plugged into the MIB510.  The mote will stream packets over the UART whenever it has power.
@n@n
   For the network of motes configuration, a base station mote needs to be programmed with TOSBase and plugged into the MIB510.  All other motes need to be installed with an XSensorMXX## application and put within range of the base station or a valid multi-hop peer.  Xlisten must then be run with the -w flag to properly parse the wireless packets.  Take care to program all the motes to the same frequency and group id.

*/

