/**
 * Handles parsing of xsensor packets.
 *
 * @file      xpacket.c
 * @author    Martin Turon
 * @version   2004/3/10    mturon      Initial version
 *
 * Copyright (c) 2004 Crossbow Technology, Inc.   All rights reserved.
 * 
 * $Id$
 */

#include "xsensors.h"

static unsigned g_datastart = XPACKET_DATASTART;

/**
 * Set the offset to the sensor data payload within the TOS packet.
 * 
 * @param     offset       Start of sensor data packet
 *
 * @author    Martin Turon
 * @version   2004/3/22       mturon      Intial version
 */
void xpacket_set_start(unsigned offset)
{
    g_datastart = offset;
}

/**
 * Converts escape sequences from a packetized TOSMsg to normal bytes.
 * 
 * @author    Martin Turon
 * @version   2004/4/07       mturon      Intial version
 */
int xpacket_depacketize(unsigned char *tos_packet)
{
    if (g_datastart > XPACKET_DATASTART) {
       	int i = 0, o = 2;    // index and offset
	while(i < XPACKET_SIZE) {
	    // Handle escape characters
	    if (tos_packet[o] == XPACKET_ESC) {
		tos_packet[i++] = tos_packet[++o] ^ 0x20;
		++o;
	    } else {
		tos_packet[i++] = tos_packet[o++];
	    }
	}
    }
}

/**
 * Returns a pointer into the packet to the data payload.
 * Also performs any required packetizer conversions if this
 * packet came from over the wireless via TOSBase. 
 * 
 * @author    Martin Turon
 * @version   2004/4/07       mturon      Intial version
 */
XbowSensorboardPacket *xpacket_get_sensor_data(unsigned char *tos_packet)
{
    return (XbowSensorboardPacket *)(tos_packet + XPACKET_DATASTART);
}

/**
 * Display a raw packet.
 * 
 * @param     packet   The TOS packet as raw bytes from the serial port
 *
 * @author    Martin Turon
 * @version   2004/3/10       mturon      Intial version
 */
void xpacket_print_raw(unsigned char *packet)
{
    int i, cnt = XPACKET_SIZE; 
    for (i=0; i<cnt; i++) {
        printf("%02x", packet[i]);
    }
    printf("\n");
}

/**
 * Display a parsed packet as exportable data -- comma delimited text.
 * 
 * @param     packet   The TOS packet as raw bytes from the serial port
 *
 * @author    Martin Turon
 * @version   2004/3/12       mturon      Intial version
 */
void xpacket_export_parsed(unsigned char *tos_packet)
{
    int i;
    uint16_t *packet = (uint16_t *)xpacket_get_sensor_data(tos_packet);
    packet += 2;  // Ignore sensorboard_id and packet_id

    for (i=0; i<8; i++) {
        if (i>0) printf(",");
        printf("%d",packet[i]);
    }
    printf("\n");
}

/**
 * Display a parsed packet as raw ADC values for each sensor on the board.
 * 
 * @param     packet   The TOS packet as raw bytes from the serial port
 *
 * @author    Martin Turon
 * @version   2004/3/10       mturon      Intial version
 */
void xpacket_print_parsed(unsigned char *tos_packet)
{
    XbowSensorboardPacket *packet = xpacket_get_sensor_data(tos_packet);

    switch(packet->packet_id) {
        case XPACKET_TEXT_MSG:
			packet->reserved2 = '\0';
            printf("msg from id=%d: %s\n", packet->node_id, packet->data);
			return;
	}

    switch(packet->sensorboard_id) {
        case XTYPE_MDA500:
            mda500_print_raw(packet);
            break;

        case XTYPE_MDA400:
            mda400_print_raw(packet);
            break;

        case XTYPE_MDA300:
            mda300_print_raw(packet);
            break;

        case XTYPE_MTS300:
            mts300_print_raw(packet);
            break;

        case XTYPE_MTS310:
            mts310_print_raw(packet);
            break;

        case XTYPE_MTS400:
            mts400_print_raw(packet);
            break;

        case XTYPE_MTS420:
            mts420_print_raw(packet);
            break;

        case XTYPE_MTS510:
            mts510_print_raw(packet);
            break;
            
        case XTYPE_MTS101:
            mts101_print_raw(packet);
            break;

        /* More sensor boards go here... */
    }
}

/**
 * Display a packet as cooked values in engineering units.
 * 
 * @param     packet   The TOS packet as raw bytes from the serial port
 *
 * @author    Martin Turon
 * @version   2004/3/11       mturon      Intial version
 */
void xpacket_print_cooked(unsigned char *tos_packet)
{
    XbowSensorboardPacket *packet = xpacket_get_sensor_data(tos_packet);

    switch(packet->packet_id) {
        case XPACKET_TEXT_MSG:
			packet->reserved2 = '\0';
            printf("MSG from id=%d: %s\n\n", packet->node_id, packet->data);
			return;
	}

    switch(packet->sensorboard_id) {
        case XTYPE_MDA500:
            mda500_print_cooked(packet);
            break;

        case XTYPE_MDA400:
            mda400_print_cooked(packet);
            break;

        case XTYPE_MDA300:
            mda300_print_cooked(packet);
            break;

        case XTYPE_MTS300:
            mts300_print_cooked(packet);
            break;

        case XTYPE_MTS310:
            mts310_print_cooked(packet);
            break;

        case XTYPE_MTS400:
            mts400_print_cooked(packet);
            break;

        case XTYPE_MTS420:
            mts420_print_cooked(packet);
            break;

        case XTYPE_MTS510:
            mts510_print_cooked(packet);
            break;
            
        case XTYPE_MTS101:
            mts101_print_cooked(packet);
            break;

        /* More sensor boards go here... */
    }
}

