/**
 * Handles conversion to engineering units of mts500 packets.
 *
 * @file      mts300.c
 * @author    Martin Turon, Hu Siquan
 * @version   2004/3/10    mturon      Initial version
 * @n         2004/4/15    husiquan    Added temp,light,accel,mic,sounder,mag
 *
 * Copyright (c) 2004 Crossbow Technology, Inc.   All rights reserved.
 * 
 * $Id$
 */
#include <math.h>
#include "../xsensors.h"

typedef struct {
	uint16_t vref;
    uint16_t thermistor;
    uint16_t light;
    uint16_t mic;
} XSensorMTS300Data;

typedef struct {
    uint16_t vref;
    uint16_t thermistor;
    uint16_t light;
    uint16_t mic;
    uint16_t accelX;
    uint16_t accelY;
    uint16_t magX;
    uint16_t magY;
} XSensorMTS310Data;


/** 
 * Converts mica2 battery reading from raw ADC data to engineering units.
 *
 * @author    Martin Turon
 *
 * To compute the battery voltage after measuring the voltage ref:
 *   BV = RV*ADC_FS/data
 *   where:
 *   BV = Battery Voltage
 *   ADC_FS = 1023
 *   RV = Voltage Reference for mica2 (1.223 volts)
 *   data = data from the adc measurement of channel 1
 *   BV (volts) = 1252.352/data
 *   BV (mv) = 1252352/data 
 *
 * Note:
 *   The thermistor resistance to temperature conversion is highly non-linear.
 *
 * @version   2004/3/29       mturon      Initial revision
 *
 */
uint16_t mts300_convert_battery(uint16_t vref) 
{
    float    x     = (float)vref;
    uint16_t vdata = (uint16_t) (1252352 / x);  
    return vdata;
}


/** 
 * Converts thermistor reading from raw ADC data to engineering units.
 *
 * @author    Martin Turon, Alan Broad
 *
 * To compute the thermistor resistance after measuring the thermistor voltage:
 * - Thermistor is a temperature variable resistor
 * - There is a 10K resistor in series with the thermistor resistor.
 * - Compute expected adc output from voltage on thermistor as: 
 *       ADC= 1023*Rthr/(R1+Rthr)
 *       where  R1 = 10K
 *              Rthr = unknown thermistor resistance
 *       Rthr = R1*ADC/(ADC_FS-ADC)
 *       where  ADC_FS = 1023
 *
 * Note:
 *   The thermistor resistance to temperature conversion is highly non-linear.
 *
 * @return    Thermistor resistance as a uint16 in unit (Ohms)
 *
 * @version   2004/3/11       mturon      Initial revision
 *
 */
uint16_t mts300_convert_thermistor_resistance(uint16_t thermistor) 
{
    float    x     = (float)thermistor;
    uint16_t vdata = 10000*x / (1023-x);
    return vdata;
}

/** 
 * Converts thermistor reading from raw ADC data to engineering units.
 *
 * @author    Martin Turon
 *
 * @return    Temperature reading from thermistor as a float in degrees Celcius
 *
 * @version   2004/3/22       mturon      Initial revision
 * @version   2004/4/19       husq      
 *
 */
float mts300_convert_thermistor_temperature(uint16_t thermistor) 
{
    float temperature, a, b, c, Rt;
    a  = 0.001307050;
    b  = 0.000214381;
    c  = 0.000000093;
    Rt = mts300_convert_thermistor_resistance(thermistor);

    temperature = 1 / (a + b * log(Rt) + c * pow(log(Rt),3));
    temperature -= 273.15;   // Convert from Kelvin to Celcius

    //printf("debug: a=%f b=%f c=%f Rt=%f temp=%f\n",a,b,c,Rt,temperature);

    return temperature;
}

/** 
 * Computes the Clairex CL94L light sensor reading 
 *
 * @author    Hu Siquan
 *
 * @return    Voltage of ADC channel as an unsigned integer in mV
 *
 * @version   2004/4/19       husq      Initial revision
 *
 */
uint16_t mts300_convert_light(uint16_t light, uint16_t vref) 
{
    float    Vbat = mts300_convert_battery(vref);
    uint16_t Vadc = (uint16_t) (light * Vbat / 1023);
    return Vadc;
}


/** 
 * Computes the ADC count of a Accelerometer - for X axis reading into 
 *  Engineering Unit (g), per calibration
 *
 * Calibration done for one test sensor - should be repeated for each unit.
 * @author    Jaidev Prabhu, Hu Siquan
 *
 * @version   2004/3/24       jdprabhu      Initial revision
 * @n         2004/4/19       husq      adaption to mts310
 *
 */
float mts310_convert_accel_x(uint16_t data)
{

    uint16_t minus_one_calibration;
    uint16_t plus_one_calibration;

    float zero_value;
    float reading;

    minus_one_calibration = 490;
    plus_one_calibration = 615;

    zero_value =  ( plus_one_calibration - minus_one_calibration ) / 2;
    reading =   (zero_value - (plus_one_calibration - data) ) / zero_value;

    return reading;
}

/** 
 * Computes the ADC count of a Accelerometer - for Y axis reading into 
 *  Engineering Unit (g), per calibration
 * Calibration done for one test sensor - should be repeated for each unit.
 *
 * @author     Jaidev Prabhu,Hu Siquan
 *
 * @version   2004/3/24       jdprabhu      Initial revision
 * @version   2004/4/19       husq      adaption to mts310
 *
 */
float mts310_convert_accel_y(uint16_t data)
{

    uint16_t minus_one_calibration;
    uint16_t plus_one_calibration;

    float zero_value;
    float reading;

    minus_one_calibration = 432;
    plus_one_calibration = 552;

    zero_value =  ( plus_one_calibration - minus_one_calibration ) / 2;
    reading =   ( zero_value - (plus_one_calibration - data) ) / zero_value;

    return reading;

}

/** 
 * Computes the ADC count of the Magnetometer - for X axis reading into 
 *  Engineering Unit (mgauss), no calibration
 *  
 *  SENSOR		Honeywell HMC1002
 *   SENSITIVITY			3.2mv/Vex/gauss
 *   EXCITATION			3.0V (nominal)
 *   AMPLIFIER GAIN		2262
 *	 ADC Input			22mV/mgauss
 *
 * @author     Hu Siquan
 *
 * @version   2004/4/26       husq      Initial Version
 *
 */
float mts310_convert_mag_x(uint16_t data,uint16_t vref)
{

 //   float    Vbat = mts300_convert_battery(vref);
 //   float Vadc = data * Vbat / 1023;
 //   return Vadc/(2.262*3.0*3.2);
 
      float Magx = data / (1.023*2.262*3.2);
      return Magx;

}

/** 
 * Computes the ADC count of the Magnetometer - for Y axis reading into 
 *  Engineering Unit (mgauss), no calibration
 *  
 *  SENSOR		Honeywell HMC1002
 *   SENSITIVITY			3.2mv/Vex/gauss
 *   EXCITATION			3.0V (nominal)
 *   AMPLIFIER GAIN		2262
 *	 ADC Input			22mV/mgauss
 *
 * @author     Hu Siquan
 *
 * @version   2004/4/26       husq      Initial Version
 *
 */
float mts310_convert_mag_y(uint16_t data,uint16_t vref)
{

   // float    Vbat = mts300_convert_battery(vref);
 //   float Vadc = (data * Vbat / 1023);
 //   return Vadc/(2.262*3.0*3.2);
      float Magy = data / (1.023*2.262*3.2);
      return Magy;

}



/** MTS300 Specific outputs of raw readings within an XBowSensorboardPacket */
void mts300_print_raw(XbowSensorboardPacket *packet) 
{
    XSensorMTS300Data *data = (XSensorMTS300Data *)packet->data;
    printf("mts300 id=%02x vref=%04x thrm=%04x light=%04x mic=%04x\n",
           packet->node_id, data->vref, data->thermistor, data->light, data->mic);
}

/** MTS300 specific display of converted readings from XBowSensorboardPacket */
void mts300_print_cooked(XbowSensorboardPacket *packet) 
{
    XSensorMTS300Data *data = (XSensorMTS300Data *)packet->data;
    printf("MTS300 [sensor data converted to engineering units]:\n"
           "   health:     node id=%i\n"
           "   battery:  = %i mv \n"
           "   temperature: =%0.2f degC\n" 
           "   light: = %i mv\n"
           "   mic: = %i ADC counts\n", 
           packet->node_id,
           mts300_convert_battery(data->vref),
           mts300_convert_thermistor_temperature(data->thermistor),
           mts300_convert_light(data->light, data->vref), data->mic);
    printf("\n");
}




/** MTS310 Specific outputs of raw readings within an XBowSensorboardPacket */
void mts310_print_raw(XbowSensorboardPacket *packet) 
{
    XSensorMTS310Data *data = (XSensorMTS310Data *)packet->data;
    printf("mts310 id=%02x vref=%4x thrm=%04x light=%04x mic=%04x\n"
           "       accelX=%04x accelY=%04x magX=%04x magY=%04x\n",
           packet->node_id, data->vref, data->thermistor, data->light, data->mic,
           data->accelX,data->accelY, data->magX, data->magY);
}

void mts310_print_cooked(XbowSensorboardPacket *packet) 
{
	
    XSensorMTS310Data *data = (XSensorMTS310Data *)packet->data;
    printf("MTS310 [sensor data converted to engineering units]:\n"
           "   health:     node id=%i\n"
           "   battery:  = %i mv \n"
           "   temperature=%0.2f degC\n" 
           "   light: = %i ADC mv\n"
           "   mic: = %i ADC counts\n"
           "   AccelX: = %f g, AccelY: = %f g\n"
           "   MagX: = %0.2f mgauss, MagY: =%0.2f mgauss\n", 
           packet->node_id,
           mts300_convert_battery(data->vref),
           mts300_convert_thermistor_temperature(data->thermistor),
           mts300_convert_light(data->light, data->vref), data->mic,
           mts310_convert_accel_x(data->accelX), mts310_convert_accel_x(data->accelY),
           mts310_convert_mag_x(data->magX,data->vref), mts310_convert_mag_y(data->magY,data->vref)
           );
    printf("\n");
  
}


