/**
 * Global definitions for Crossbow sensor boards.
 *
 * @file      xsensors.h
 * @author    Martin Turon
 * @version   2004/3/10    mturon      Initial version
 *
 * Copyright (c) 2004 Crossbow Technology, Inc.   All rights reserved.
 * 
 * $Id$
 */

#ifndef __XSENSORS_H__
#define __XSENSORS_H__

#include <stdio.h>

/** 
 *  A unique identifier for each Crossbow sensorboard. 
 *
 *  Note: The sensorboard id is organized to allow for identification of
 *        host mote as well:
 *
 *  if  (sensorboard_id < 0x80)  // mote is a mica2dot
 *  if  (sensorboard_id > 0x7E)  // mote is a mica2
 *
 * @version   2004/3/10    mturon      Initial version
 */
typedef enum {
  // mica2dot sensorboards 
  XTYPE_MDA500 = 0x01,   
  XTYPE_MTS510,

  // mica2 sensorboards 
  XTYPE_MDA400 = 0x80,   
  XTYPE_MDA300,
  XTYPE_MTS101,
  XTYPE_MTS300,
  XTYPE_MTS310,
  XTYPE_MTS400,
  XTYPE_MTS420,
} XbowSensorboardType;

/** 
 * Reserves general packet types that xlisten handles for all sensorboards.
 *
 * @version      2004/4/2     mturon      Initial version
 */
typedef enum {
  // reserved packet ids 
  XPACKET_ESC      = 0x7D,    //!< Reserved for serial packetizer escape code.
  XPACKET_START    = 0x7E,    //!< Reserved for serial packetizer start code.
  XPACKET_TEXT_MSG = 0xF8,    //!< Special id for sending text error messages.
} XbowGeneralPacketType;

/** Encodes sensor readings into the data payload of a TOS message. */
typedef struct {
    uint8_t  sensorboard_id;
    uint8_t  packet_id;
    uint8_t  node_id;
    uint8_t  reserved;
    uint16_t data[12];
    uint8_t  reserved2;
} XbowSensorboardPacket;

#define XPACKET_SIZE               36
#define XPACKET_GROUP               3  //!< offset to group id of TOS packet
#define XPACKET_LENGTH              4  //!< offset to length of TOS packet
#define XPACKET_DATASTART           5  //!< UART offset to data payload
#define XPACKET_DATASTART_WIRELESS  7  //!< Wireless offset to data payload

/* Sensorboard data packet definitions */
void xpacket_print_raw     (unsigned char *tos_packet);
void xpacket_print_parsed  (unsigned char *tos_packet);
void xpacket_print_cooked  (unsigned char *tos_packet);
void xpacket_export_parsed (unsigned char *tos_packet);
int  xpacket_depacketize   (unsigned char *tos_packet);
void xpacket_set_start     (unsigned offset);

/* Serial port routines. */
int xserial_port_open ();
int xserial_port_dump ();
int xserial_port_sync_packet (int serline);
int xserial_port_read_packet (int serline, unsigned char *buffer, int cnt);

unsigned xserial_set_baudrate (unsigned baudrate);
unsigned xserial_set_baud     (const char *baud);
void     xserial_set_device   (const char *device);
void     xserial_set_verbose  (int verbose);

/* Sensorboard specific conversion routines. */
/* From boards/mda500.c */
void mda500_print_raw    (XbowSensorboardPacket *packet);
void mda500_print_cooked (XbowSensorboardPacket *packet);
void mda400_print_raw    (XbowSensorboardPacket *packet);
void mda400_print_cooked (XbowSensorboardPacket *packet);

/* From boards/mda300.c */
void mda300_print_raw    (XbowSensorboardPacket *packet);
void mda300_print_cooked (XbowSensorboardPacket *packet);

/* From boards/mts300.c */
void mts300_print_raw    (XbowSensorboardPacket *packet);
void mts300_print_cooked (XbowSensorboardPacket *packet);
void mts310_print_raw    (XbowSensorboardPacket *packet);
void mts310_print_cooked (XbowSensorboardPacket *packet);

/* From boards/mts400.c */
void mts400_print_raw    (XbowSensorboardPacket *packet);
void mts400_print_cooked (XbowSensorboardPacket *packet);
void mts420_print_raw    (XbowSensorboardPacket *packet);
void mts420_print_cooked (XbowSensorboardPacket *packet);

/* From boards/mts510.c */
void mts510_print_raw    (XbowSensorboardPacket *packet);
void mts510_print_cooked (XbowSensorboardPacket *packet);

/* From boards/mts101.c */
void mts101_print_raw    (XbowSensorboardPacket *packet);
void mts101_print_cooked (XbowSensorboardPacket *packet);


#endif  /* __SENSORS_H__ */



