/*
 * Copyright (c) 2004, Technische Universität Berlin
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright 
 *   notice, this list of conditions and the following disclaimer in the 
 *   documentation and/or other materials provided with the distribution.
 * - Neither the name of the Technische Universität Berlin nor the names 
 *   of its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED 
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, 
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * - Description ----------------------------------------------------------
 * Mapping of ADC-interface to msp-specific interface / components.
 * Read ADC_README about how to use the provided
 * interfaces, e.g. ADCControl.bind has a slightly different meaning.
 * - Revision -------------------------------------------------------------
 * $Revision$
 * $Date$
 * @author Vlado Handziski
 * @author Jan Hauer 
 * ========================================================================
 */
includes MSP430ADC12;

module ADCM
{
  provides {
    interface ADC[uint8_t port];
    interface ADCControl;
  }
  uses {
    interface MSP430ADC12External as ExternalADC[uint8_t port];
    interface MSP430ADC12Internal as InternalADC;
  }
}

implementation
{
  uint8_t samplingRate;
  uint16_t continousData;
  bool initialized = 0;

  command result_t ADCControl.init() {
    if (!initialized){
      samplingRate = 0xFF;  // max. sampling time (1024 ticks ADC12CLK)
      call ExternalADC.setMaxExtInterfaces[0](8);
      continousData = 0;
      initialized = 1;
      return SUCCESS;
    } else return FAIL;
  }

  command result_t ADCControl.setSamplingRate(uint8_t rate) {
    // Assumption: SMCLK runs at 1 MHz
    switch (rate){
      case (TOS_ADCSample3750ns): samplingRate = 0x00; break;
      case (TOS_ADCSample7500ns): samplingRate = 0x01; break;
      case (TOS_ADCSample15us):   samplingRate = 0x02; break;
      case (TOS_ADCSample30us):   samplingRate = 0x03; break;
      case (TOS_ADCSample60us):   samplingRate = 0x04; break;
      case (TOS_ADCSample120us):  samplingRate = 0x06; break;
      case (TOS_ADCSample240us):  samplingRate = 0x08; break;
      case (TOS_ADCSample480us):  samplingRate = 0x0A; break;
    }
    return SUCCESS;
  }

  command result_t ADCControl.bindPort(uint8_t port, uint8_t adcPort) {
    MSP430ADC12StandardSettings_t settings;
    if (port < 8){
      settings.externalChannel = port;
      settings.referenceVoltage = adcPort;
      settings.sampleHoldTime = samplingRate & 0x0F;
      return call ExternalADC.bind[port](settings);
    } else
      return SUCCESS;  // internal channel
  }
  
  async command result_t ADC.getData[uint8_t port](){
    uint16_t mask = 1;
    result_t result;
    mask <<= port;
    continousData &= ~mask;
    if (port < 8){
      call ExternalADC.setSamplingTime[port](samplingRate);
      result = call ExternalADC.getSingleData[port]();
    } else 
      result = call InternalADC.getData(port);
    return result;
  }
  
  async command result_t ADC.getContinuousData[uint8_t port](){
    uint16_t mask = 1;
    result_t result;
    mask <<= port;
    continousData |= mask;
    if (port < 8){
      call ExternalADC.setSamplingTime[port](samplingRate);
      result = call ExternalADC.getSingleData[port]();
    } else 
      result = call InternalADC.getData(port);
    return result;
  }
  
  event void ExternalADC.dataReadySingle[uint8_t port](uint16_t data){
    uint16_t mask = 1;
    result_t result;
    mask <<= port;
    result = signal ADC.dataReady[port](data);
    if (result == SUCCESS && continousData & mask)
      call ExternalADC.getSingleData[port]();
  }
  
  event void InternalADC.dataReady(MSP430ADC12InternalChannel_t channel, uint16_t data){
    uint16_t mask = 1;
    result_t result;
    mask <<= channel;
    result = signal ADC.dataReady[channel](data);
    if (result == SUCCESS && continousData & mask)
      call InternalADC.getData(channel);
  }

  default async event result_t ADC.dataReady[uint8_t port](uint16_t data) {
    return FAIL;
  }
 
  event void ExternalADC.dataReadySequence[uint8_t port](){}
  async event void ExternalADC.dataReadySingleRpt[uint8_t port](uint16_t data){}
  async event void ExternalADC.dataReadySequenceRpt[uint8_t port](){}
  
  
}



