#ifndef MSP430ADC12_H
#define MSP430ADC12_H

/* The following 2 macros define the default settings for the ADC12
   control registers when used with the MSP430ADC12External interface 
   (or when ADCC is used).
   ADC12CTL0: 
   - multiple sample and conversion triggered by rising edge of SHI
   - NOTE: REFON and REF2_5V are ignored in MSP430ADCM.nc, use RefVolt component
   ADC12CTL1: 
   - clock select: SMCLK
   - clock divider: 1
   - sample-input signal not inverted
   - SAMPCON signal is sourced from the sampling timer (SHP=1)
   - sample-and-hold source: ADC12SC bit
*/
#define ADC12CTL0_DEFAULT 0x0000
#define ADC12CTL1_DEFAULT 0x0218

/* The following 2 macros define the settings for the ADC12
   control registers when used with the MSP430ADC12Internal interface.
   ADC12CTL0: 
   - Sample-and-hold-time is 64 ticks of AC12CLK, with SMCL=1MHz thats 64us.
   - NOTE: REFON and REF2_5V are ignored in MSP430ADCM.nc, use RefVolt component
   - ADC on
   ADC12CTL1: 
   - clock select: SMCLK
   - clock divider: 1
   - sample-input signal not inverted
   - SAMPCON signal is sourced from the sampling timer (SHP=1)
   - sample-and-hold source: ADC12SC bit
   - conversion start address is ADC12MEM8
*/
#define ADC12CTL0_INTERNAL 0x4070
#define ADC12CTL1_INTERNAL 0x8218

#ifndef EXT_IF_INSTANCES
#if NESC >= 110
#define EXT_IF_INSTANCES (uniqueCount("MSP430ADC12External")>8?8:uniqueCount("MSP430ADC12External"))
#else
#define EXT_IF_INSTANCES 8
#endif
#endif

typedef struct 
{
  unsigned int externalChannel: 3;     // input channel [0..7]
                                        //   0  ->  A0
                                        //   1  ->  A1
                                        //   2  ->  A2
                                        //   3  ->  A3
                                        //   4  ->  A4
                                        //   5  ->  A5
                                        //   6  ->  A6
                                        //   7  ->  A7
  
  unsigned int referenceVoltage: 3;    // reference voltage [0..7], use referenceVoltage_enum
  unsigned int sampleHoldTime: 4;      // sample-hold-time (sample duration), use sampleHold_enum
                              // (clock defined in ADC12CTL1_DEFAULT)
} MSP430ADC12StandardSettings_t;


typedef struct 
{
  unsigned int externalChannel: 3;     // input channel [0..7]
                                        //   0  ->  A0
                                        //   1  ->  A1
                                        //   2  ->  A2
                                        //   3  ->  A3
                                        //   4  ->  A4
                                        //   5  ->  A5
                                        //   6  ->  A6
                                        //   7  ->  A7
  
  unsigned int referenceVoltage: 3;    // reference voltage [0..7], use referenceVoltage_enum
  unsigned int sampleHoldTime: 4;      // sample-hold-time (sample duration), use sampleHold_enum
  unsigned int clockSource: 2;         // ADC12 clock source, use clockSource_enum
  unsigned int clockDiv: 3;            // ADC12 clock divider, use clockDiv_enum
  unsigned int sampleHoldSource: 2;    // source (timer) select for the SHI signal, use enum sampleHoldSource_enum
                                       // make sure this timer is not used by another component !
} MSP430ADC12AdvancedSettings_t;

typedef enum
{
   ADC_FAIL = 0,
   ADC_SUCCESS = 1,
   ADC_QUEUED = 2
} adcResult_t;

typedef enum
{
  EXTERNAL_REFERENCE = 8,          // VeREF+ (input channel 8)
  REFERENCE_VOLTAGE_RATIO = 9,     // VREF-/VeREF- (input channel 9)
  INTERNAL_TEMPERATURE = 10,       // Temperature diode (input channel 10)
  BATTERY_LEVEL = 11               // (AVcc-AVss)/2 (input channel 11-15)
} MSP430ADC12InternalChannel_t;

enum sampleHoldSource_enum
{
  HOLDSOURCE_TIMERA_OUT1 = 1,    // TimerA.CompareA1
  HOLDSOURCE_TIMERB_OUT0 = 2,    // TimerB.CompareB0
  HOLDSOURCE_TIMERB_OUT1 = 3     // TimerB.CompareB1
};

enum clockDiv_enum
{
   CLOCK_DIV_1 = 0,                // ADC12 clock divider of 1
   CLOCK_DIV_2 = 1,                // ADC12 clock divider of 2
   CLOCK_DIV_3 = 2,                // ADC12 clock divider of 3
   CLOCK_DIV_4 = 3,                // ADC12 clock divider of 4
   CLOCK_DIV_5 = 4,                // ADC12 clock divider of 5
   CLOCK_DIV_6 = 5,                // ADC12 clock divider of 6
   CLOCK_DIV_7 = 6,                // ADC12 clock divider of 7
   CLOCK_DIV_8 = 7,                // ADC12 clock divider of 8
};

enum referenceVoltage_enum
{
   REFERENCE_AVcc_AVss = 0,                  // VR+ = AVcc   and VR-= AVss
   REFERENCE_VREFplus_AVss = 1,              // VR+ = VREF+  and VR-= AVss
   REFERENCE_VeREFplus_AVss = 2,             // VR+ = VeREF+ and VR-= AVss
   REFERENCE_AVcc_VREFquotient = 4,          // VR+ = AVcc   and VR-= VREF-/VeREF- 
   REFERENCE_VREFplus_VREFquotient = 5,      // VR+ = VREF+  and VR-= VREF-/VeREF-   
   REFERENCE_VeREFplus_VREFquotient = 6      // VR+ = VeREF+ and VR-= VREF-/VeREF-
};

enum clockSource_enum
{
   CLOCK_SOURCE_ADC12OSC = 0,                // ADC12OSC
   CLOCK_SOURCE_ACLK = 1,                    // ACLK
   CLOCK_SOURCE_MCLK = 2,                    // MCLK
   CLOCK_SOURCE_SMCLK = 3                    // SMCLK
};

enum sampleHold_enum
{
   SAMPLE_HOLD_4_CYCLES = 0,         // sampling duration is 4 clock cycles
   SAMPLE_HOLD_8_CYCLES = 1,         // ...
   SAMPLE_HOLD_16_CYCLES = 2,         
   SAMPLE_HOLD_32_CYCLES = 3,         
   SAMPLE_HOLD_64_CYCLES = 4,         
   SAMPLE_HOLD_96_CYCLES = 5,        
   SAMPLE_HOLD_123_CYCLES = 6,        
   SAMPLE_HOLD_192_CYCLES = 7,        
   SAMPLE_HOLD_256_CYCLES = 8,        
   SAMPLE_HOLD_384_CYCLES = 9,        
   SAMPLE_HOLD_512_CYCLES = 10,        
   SAMPLE_HOLD_768_CYCLES = 11,        
   SAMPLE_HOLD_1024_CYCLES = 12       
};

/////////////////////////////////////////////////////

typedef enum
{
  TIMERA_OUT1 = 1,    // TimerA.CompareA1
  TIMERB_OUT0 = 2,    // TimerB.CompareB0
  TIMERB_OUT1 = 3     // TimerB.CompareB1
} MSP430ADC12Timer;


typedef enum
{
  SINGLE_CHANNEL_SINGLE_CONVERSION = 0,
  SEQUENCE_OF_CHANNELS = 1,
  REPEAT_SINGLE_CHANNEL = 2,
  REPEAT_SEQUENCE_OF_CHANNELS = 3,
  INTERNAL_CHANNEL = 4,
  ADVANCED_SEQUENCE_OF_CHANNELS = 5,
  ADVANCED_REPEAT_SINGLE_CHANNEL = 6,
  ADVANCED_REPEAT_SEQUENCE_OF_CHANNELS = 7,
  ADC_IDLE = 8
} MSP430ADC12ConversionMode_t;

typedef struct 
{
  volatile unsigned
  inch: 4,            // input channel
  sref: 3,            // reference voltage
  eos: 1;             // end of sequence flag
} __attribute__ ((packed)) adc12memctl_t;

#endif
