/*
 * Copyright (c) 2004, Technische Universitaet Berlin
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - Neither the name of the Technische Universitaet Berlin nor the names
 *   of its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * - Description ----------------------------------------------------------
 * This interface can be used, if an external input channel (INCH=[A0..A7]) 
 * is the source of conversion and the start of sampling should be triggered
 * by TimerA or TimerB. The advantage (compared to MSP430ADC12External) is 
 * that it is now possible, to specify an interval which allows you to 
 * e.g. sample a channel every x clock cycles (of TimerA/B clock source)
 * and - if used in sequence mode - signal one event after n samples
 * have been taken.
 * The source for sampling start can be Timer_A.OUT1, Timer_B.OUT0 or
 * Timer_B.OUT1. Therefore the according interfaces of MSP430TimerM are
 * used (CompareA1, CompareB0, CompareB1) - make sure no one else uses it.
 *
 * NOTE: It is REQUIRED that each interface is instantiated with
 *       unique("MSP430ADC12Advanced"). 
 *
 * - Revision -------------------------------------------------------------
 * $Revision$
 * $Date$
 * @author: Jan Hauer (hauer@tkn.tu-berlin.de)
 * ========================================================================
 */
        
includes  MSP430ADC12;

interface MSP430ADC12Advanced
{     
  /**
   * Binds this interface instance to the values specified in
   * <code>settings</code>. This command must be called once, 
   * before any call to the <code>get...</code>-commands
   * will be successful.
   *
   * @return FAIL if interface parameter is out of bounds (see NOTE),
   *         SUCCESS otherwise
   */
  command result_t bind(MSP430ADC12AdvancedSettings_t settings);
       
  /**
   * Initiates a single conversion in repeat mode. The event 
   * <code>dataReadySingleRepeat</code> will be signalled after each 
   * conversion has finished. A conversion is made each
   * <code>jiffies</code> clock ticks, the first conversion starts at
   * once. The clock source for this interface instance was set 
   * in <code>bind</code> in <code>settings.shs</code>. 
   * This process only stops if the <code>stop</code> command is called.
   * Until then the ADC is blocked for other requests.
   *
   * @param jiffies clock ticks between samples (clock source specified 
   *                in <code>bind</code>)
   * @return FAIL if interface parameter is is out of bounds (see NOTE),
   *              or ADC is already in use
   *         SUCCESS otherwise
   */
  command result_t getSingleDataRepeat(uint16_t jiffies);
  
  /**
   * Initiates a sequence of conversions in single mode. All results
   * will be written to the specified array and after the last result
   * <code>dataReady</code> will be signalled. A conversion is made each
   * <code>jiffies</code> clock ticks. The clock source for this
   * interface instance was set in <code>bind</code> 
   * in <code>settings.shs</code>. 
   * The maximum length for a sequence can be determined by the
   * <code>getMaxSequenceSize()</code> command, it will be [8..16].
   * If the ADC is currently busy the call is rejected.
   *
   * @param jiffies clock ticks between samples (clock source specified 
   *                in <code>bind</code>)
   * @param dataDest array for the result to be stored.
   * @param length the length of the array and size of sequence
   *
   * @return FAIL if interface parameter is not is out of bounds (see NOTE),
   *              or ADC is busy or length is out of bounds
   *         SUCCESS otherwise
   */
  command result_t getSequenceData(uint16_t dataDest[], uint8_t length, uint16_t jiffies);
  
  /**
   * Initiates a sequence of conversions in repeat mode. The event 
   * <code>dataReady</code> will be signalled after each sequence of
   * conversions has finished. A conversion is made each
   * <code>jiffies</code> clock ticks. The clock source for this
   * interface instance was set in <code>bind</code> 
   * in <code>settings.shs</code>. 
   * This process only stops if the <code>stop</code> command is called.
   * The maximum length for a sequence can be determined by the
   * <code>getMaxSequenceSize()</code> command, it will be [8..16].
   * If the ADC is currently busy the call is rejected.
   *
   * @param dataDest array for the result to be stored.
   * @param length the length of the array and size of sequence
   *
   * @return FAIL if interface parameter is not is out of bounds (see NOTE),
   *              or ADC is busy or length is out of bounds
   *         SUCCESS otherwise
   */
  command result_t getSequenceDataRepeat(uint16_t dataDest[], uint8_t length, uint16_t jiffies);
   
  /*
   * Stops any further conversions for this interface instance.
   * Conversions already started (hardware) are finished and one last event
   * will be signalled upon completion (and until then no call to e.g. 
   * <code>getSequenceData</code> will be successful).
   */
  async command void stop();
  
  /*
   * Returns the maximum size for a sequence of conversions, it will be 
   * minimum 8, maximum 16.
   *
   * @return maximum number for a sequence of conversions
   */
  command uint8_t getMaxSequenceSize();
  
  /*
   * Single conversion data from repeated mode is ready.
   * (This event is emitted in interrupt context!)
   * @param data The lower 12 bits are the conversion result.
   */
  async event void dataReadySingleRpt(uint16_t data);
    
  /*
   * Sequence conversion data is ready and stored at the declared 
   * memory location.
   */
  event void dataReadySequence();

  /*
   * Sequence conversion data from repeated mode is ready and 
   * stored at the declared memory location.
   * (This event is emitted in interrupt context!)
   */
  async event void dataReadySequenceRpt();
}

