/*
 * Copyright (c) 2004, Technische Universitaet Berlin
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 * - Neither the name of the Technische Universitaet Berlin nor the names
 *   of its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * - Description ----------------------------------------------------------
 * This interface can be used, if an external input channel (INCH=[A0..A7]) 
 * is the source of conversion. Internal input channels, such as VeREF+ or
 * internal temperature sensor, can not be used with this interface.
 * Before the first conversion, <code>bind()</code> needs to be called to
 * associate the interface instance with some settings.
 * All settings not specified in <code>bind()</code> will be the default 
 * settings as specified in MSP430ADC12.h.
 *
 * NOTE: It is RECOMMENDED that each interface is instantiated with
 *       unique("MSP430ADC12External"). 
 *       (see also <code>setMaxExtInterfaces()</code>)
 *
 * - Revision -------------------------------------------------------------
 * $Revision$
 * $Date$
 * @author: Jan Hauer (hauer@tkn.tu-berlin.de)
 * ========================================================================
 */
        
includes  MSP430ADC12;

interface MSP430ADC12External
{ 
  /**
   * Binds this interface instance to the values specified in
   * <code>settings</code>. This command must be called once, 
   * before any call to the <code>get...</code>-commands
   * can be made.
   *
   * @return FAIL if interface parameter is out of bounds (see NOTE),
   *         SUCCESS otherwise
   */
  command result_t bind(MSP430ADC12StandardSettings_t settings);
    
  /**
   * Returns the <code>settings</code> this interface instance was bound to.
   *
   * @return settings for this interface instance
   */
  command MSP430ADC12StandardSettings_t getSettings();
  
  /**
   * Sets the sampling time used for this interface instance.
   * This overwrites the value specified in <code>bind</code>.
   *
   * @param time The lower 4 bits are interpreted as SHTx (sample-hold-time)
   */
  command void setSamplingTime(uint8_t time);
  
  /**
   * Initiates one single conversion. The event <code>dataReadySingle</code> 
   * will be signalled after completion.
   * If the ADC is currently busy, the command is buffered
   * (only once). Buffered commands are served when the ADC is
   * available in round-robin fashion.
   *
   * @return ADC_FAIL if interface parameter is out of bounds (see NOTE) or
   *                  request is already buffered
   *         ADC_SUCCESS if conversion triggered
   *         ADC_QUEUED  if request queued
   */
  command adcResult_t getSingleData(); 
   
  /**
   * Initiates a single conversion in repeat mode. The event 
   * <code>dataReadySingleRepeat</code> will be signalled after each 
   * conversion has finished. The next conversion is always triggered 
   * immediately after the previous one has completed (if you need
   * a delay in between use <code>MSP430ADC12Advanced</code> interface). 
   * This process only stops if the <code>stop</code> command is called.
   * Until then the ADC is blocked for other requests.
   *
   * @return FAIL if interface parameter is is out of bounds (see NOTE),
   *              or ADC is already in use
   *         SUCCESS if conversion triggered
   */
  command result_t getSingleDataRepeat(); 
  
  /**
   * Initiates a sequence of conversions in single mode. All results
   * will be written to the specified array and after the last result
   * <code>dataReady</code> will be signalled.
   * The maximum length for a sequence can be determined by the
   * <code>getMaxSequenceSize()</code> command, it will be [8..16].
   * If the ADC is currently busy the call is rejected.
   *
   * @param dataDest array for the result to be stored.
   * @param length the length of the array and size of sequence
   *
   * @return FAIL if interface parameter is not is out of bounds (see NOTE),
   *              or ADC is busy or length is out of bounds
   *         SUCCESS if conversion triggered
   */
  command result_t getSequenceData(uint16_t dataDest[], uint8_t length);
  
  /**
   * Initiates a sequence of conversions in repeat mode. The event 
   * <code>dataReady</code> will be signalled after each sequence of
   * conversions has finished. The next conversion is always triggered 
   * immediately after the previous one has completed (if you need
   * a delay in between use <code>MSP430ADC12Advanced</code> interface).
   * This process only stops if the <code>stop</code> command is called.
   * The maximum length for a sequence can be determined by the
   * <code>getMaxSequenceSize()</code> command, it will be [8..16].
   * If the ADC is currently busy the call is rejected.
   *
   * @param dataDest array for the result to be stored.
   * @param length the length of the array and size of sequence
   *
   * @return FAIL if interface parameter is not is out of bounds (see NOTE),
   *              or ADC is busy or length is out of bounds
   *         SUCCESS if conversion triggered
   */
  command result_t getSequenceDataRepeat(uint16_t dataDest[], uint8_t length);
   
  /*
   * Stops any further conversions for this interface instance.
   * Conversions already triggered in hardware will be cancelled, thus
   * after this call returns there will be no <code>dataReady...</code>
   * events signalled for this interface instance (unless, of course, a new 
   * call to the <code>get...</code>-commands is made).
   */
  async command void stop();
  
  /*
   * Returns the maximum size for a sequence of conversions, it will be 
   * minimum 8, maximum 16.
   *
   * @return maximum number for a sequence of conversions
   */
  command uint8_t getMaxSequenceSize();

  /*
   * Sets the maximum number of external interfaces to be reserved for
   * converting external channels. This overwrites the internal representation of
   * uniqueCount("MSP430ADC12External"), so it might be fatal :)
   * Only use it, if you know what you're doing.
   *
   * @param max Maximum number of external interfaces supported
   */
  command void setMaxExtInterfaces(uint8_t max);
  
  /*
   * Single conversion data is ready.
   * @param data The lower 12 bits are the conversion result.
   */
  event void dataReadySingle(uint16_t data);
  
  /*
   * Single conversion data from repeated mode is ready.
   * (This event is emitted in interrupt context!)
   * @param data The lower 12 bits are the conversion result.
   */
  async event void dataReadySingleRpt(uint16_t data);
    
  /*
   * Sequence conversion data is ready and stored at the declared 
   * memory location.
   */
  event void dataReadySequence();

  /*
   * Sequence conversion data from repeated mode is ready and 
   * stored at the declared memory location.
   * (This event is emitted in interrupt context!)
   */
  async event void dataReadySequenceRpt();
}

