/*
 * Copyright (c) 2004, Technische Universitaet Berlin
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright 
 *   notice, this list of conditions and the following disclaimer in the 
 *   documentation and/or other materials provided with the distribution.
 * - Neither the name of the Technische Universitaet Berlin nor the names 
 *   of its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED 
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, 
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * - Description ----------------------------------------------------------
 * Wrapper around lowlevel ADC component.
 * Different ports can be converted without interfering each other
 * (there is no ADCControl).
 * After an interface is bound to a channel, MSP430ADC12M takes care
 * that the according settings are used each time a conversion is requested
 * by that interface instance. 
 * - Revision -------------------------------------------------------------
 * $Revision$
 * $Date$
 * @author: Jan Hauer (hauer@tkn.tu-berlin.de)
 * ========================================================================
 */
includes MSP430ADC12;

module MSP430ADC12M {
  provides interface StdControl;
  provides interface MSP430ADC12External[uint8_t id];
  provides interface MSP430ADC12Internal;
  provides interface MSP430ADC12Advanced[uint8_t id];
  
  uses interface HPLADC12;
  
  uses interface MSP430Compare as CompareA1;
  uses interface MSP430Compare as CompareB0;
  uses interface MSP430Compare as CompareB1;
}
implementation
{  
  norace MSP430ADC12ConversionMode_t cmode; // current conversion mode 
  uint32_t samplingTime;                    // 4 bits for each channel 0..7 and its sampling time
  uint8_t queuedRequests;                   // one bit per ext. interface and queued request 
  uint8_t queueOffset;                      // offset into queue to ensure round-robin 
  norace uint16_t *seqResultPtr;            // mem-location to store sequence conversion results
  uint8_t owner;                            // interface instance that conversion is done for
  norace uint8_t maxExtIf;                  // maximum external interfaces
                             // advancedSettings store detailed settings (if any) 
  MSP430ADC12AdvancedSettings_t advancedSettings[uniqueCount("MSP430ADC12Advanced")];
  norace uint16_t advancedInterval;         // time interval between conversions for advanced interface
  norace MSP430ADC12Timer advancedTimer;    // timer used for SHI signal
  norace bool cancelled;                    // TRUE if stop command was called
  
  task void checkQueue();
  
  command result_t StdControl.init(){ 
    uint16_t ctl0 = ADC12CTL0_DEFAULT;
    uint16_t ctl1 = ADC12CTL1_DEFAULT;
    maxExtIf = EXT_IF_INSTANCES;
    cmode = ADC_IDLE;
    cancelled = FALSE;
    call HPLADC12.disableConversion();
    call HPLADC12.setControl0_IgnoreRef(*(adc12ctl0_t*) &ctl0);
    call HPLADC12.setControl1(*(adc12ctl1_t*) &ctl1);
    call HPLADC12.setIEFlags(0x0000);       
    return SUCCESS;
  }
  
  command result_t StdControl.start(){ 
    return SUCCESS; 
  }
  
  command result_t StdControl.stop(){
    call HPLADC12.disableConversion();
    call HPLADC12.setIEFlags(0x0000);
  }
    
  /**************************** MSP430ADC12External ***********************************/
  
  command result_t MSP430ADC12External.bind[uint8_t num](MSP430ADC12StandardSettings_t settings){
    adc12memctl_t memctrlword = {  inch: settings.externalChannel,
                                   sref: settings.referenceVoltage,
                                   eos: 0 };
    uint32_t tmpSHT = settings.sampleHoldTime;
    
    if (num < maxExtIf) {
      // dedicate ADC12MCTL[num] to this interface instance
      tmpSHT <<= num * 4;
      call HPLADC12.setMemControl(num, memctrlword);
      atomic samplingTime |= tmpSHT; 
    } else return FAIL;
    return SUCCESS;
  }
  
  command MSP430ADC12StandardSettings_t MSP430ADC12External.getSettings[uint8_t num](){
    MSP430ADC12StandardSettings_t settings;
    adc12memctl_t memctl = call HPLADC12.getMemControl(num);
    settings.externalChannel = memctl.inch;
    settings.referenceVoltage = memctl.sref;
    settings.sampleHoldTime = ((uint8_t) samplingTime >> num*4) & 0x0F;
    return settings;
  }

  command void MSP430ADC12External.setSamplingTime[uint8_t num](uint8_t time){    
    uint8_t *sTimePtr = (uint8_t*) &samplingTime;
    uint8_t tempTime;
    sTimePtr += num / 2;
    tempTime = *sTimePtr;
    if (num % 2){
      time <<= 1;
      tempTime &= ~0xF0;
      tempTime |= (time & 0xF0);
    } else {
      tempTime &= ~0x0F;
      tempTime |= (time & 0x0F);
    }
    *sTimePtr = tempTime;    
  }
  
    
  command adcResult_t MSP430ADC12External.getSingleData[uint8_t num](){
    uint8_t mask = 1;
    bool gotIt = FALSE;
    uint8_t queuedAlready;
    
    if (num < maxExtIf){
      mask <<= num;
      atomic {
        queuedAlready = queuedRequests & mask;
        queuedRequests |= mask;
      }
      if (queuedAlready)
        return ADC_FAIL;
      atomic {
        if (cmode == ADC_IDLE){
          cmode = SINGLE_CHANNEL_SINGLE_CONVERSION;
          owner = num;
          gotIt = TRUE;
        }
      }
      if (gotIt){ // trigger conversion
        call HPLADC12.disableConversion();
        call HPLADC12.on();
        call HPLADC12.setConversionMode(SINGLE_CHANNEL_SINGLE_CONVERSION, num);
        call HPLADC12.setSHT((uint8_t) (samplingTime >> num*4));
        call HPLADC12.setIEFlags(mask);
        call HPLADC12.startConversion();
        return ADC_SUCCESS;
      } 
    } else // num too big
        return ADC_FAIL;
    return ADC_QUEUED;
  } 
  
  command result_t MSP430ADC12External.getSingleDataRepeat[uint8_t num](){
    uint16_t mask = 1;
    bool gotIt = FALSE;
       
    if (num >= maxExtIf)
      return FAIL;
    atomic {
      if (cmode == ADC_IDLE){
        cmode = REPEAT_SINGLE_CHANNEL;
        owner = num;
        gotIt = TRUE;
      }
    }
    if (gotIt){
      mask <<= num;
      call HPLADC12.disableConversion();
      call HPLADC12.on();
      call HPLADC12.setConversionMode(REPEAT_SINGLE_CHANNEL, num);
      call HPLADC12.setSHT((uint8_t) (samplingTime >> num*4));
      call HPLADC12.setIEFlags(mask);
      call HPLADC12.startConversion();
    } else 
      return FAIL;
    return SUCCESS;  
  }
  
  command result_t MSP430ADC12External.getSequenceData[uint8_t num](uint16_t *dataDest, uint8_t length){
    uint8_t i, maxseqSize, sampletime;
    uint16_t mask = 1;
    bool gotIt = FALSE;
    
    maxseqSize = call MSP430ADC12External.getMaxSequenceSize[0]();
    if (num >= maxExtIf || !length || length > maxseqSize)
      return FAIL;
    atomic {
      if (cmode == ADC_IDLE){
        cmode = SEQUENCE_OF_CHANNELS;
        owner = num;
        gotIt = TRUE;
      }
    }
    if (gotIt){ // trigger conversion
      adc12memctl_t memctl = call HPLADC12.getMemControl(num);
      memctl.eos = 0;
      seqResultPtr = dataDest;
      call HPLADC12.disableConversion();
      for (i=maxExtIf; i<maxExtIf+length-1; i++)
        call HPLADC12.setMemControl(i,memctl);
      memctl.eos = 1;  
      call HPLADC12.setMemControl(i,memctl);
      call HPLADC12.on();
      call HPLADC12.setConversionMode(SEQUENCE_OF_CHANNELS, maxExtIf);
      sampletime = (uint8_t) samplingTime >> num*4 & 0x00FF;
      sampletime |= sampletime << 4;
      call HPLADC12.setSHT((uint8_t) (samplingTime >> num*4));
      call HPLADC12.setIEFlags(mask << (maxExtIf+length-1));
      call HPLADC12.startConversion();
    } else
        return FAIL;
    return SUCCESS;
  }
  
  command result_t MSP430ADC12External.getSequenceDataRepeat[uint8_t num](uint16_t *dataDest, uint8_t length){
    uint8_t i, maxseqSize, sampletime;
    uint16_t mask = 1;
    bool gotIt = FALSE;
    
    maxseqSize = call MSP430ADC12External.getMaxSequenceSize[0]();
    if (num >= maxExtIf || !length || length > maxseqSize)
      return FAIL;
    atomic {
      if (cmode == ADC_IDLE){
        cmode = REPEAT_SEQUENCE_OF_CHANNELS;
        owner = num;
        gotIt = TRUE;
      }
    }
    if (gotIt){ // trigger conversion
      adc12memctl_t memctl = call HPLADC12.getMemControl(num);
      memctl.eos = 0;
      seqResultPtr = dataDest;
      call HPLADC12.disableConversion();
      for (i=maxExtIf; i<maxExtIf+length-1; i++)
        call HPLADC12.setMemControl(i,memctl);
      memctl.eos = 1;  
      call HPLADC12.setMemControl(i,memctl);
      call HPLADC12.on();
      call HPLADC12.setConversionMode(REPEAT_SEQUENCE_OF_CHANNELS, maxExtIf);
      sampletime = (uint8_t) samplingTime >> num*4 & 0x00FF;
      sampletime |= sampletime << 4;
      call HPLADC12.setSHT((uint8_t) (samplingTime >> num*4));
      call HPLADC12.setIEFlags(mask << (maxExtIf+length-1));
      call HPLADC12.startConversion();
    } else
        return FAIL;
    return SUCCESS;  
  }
    

  async command void MSP430ADC12External.stop[uint8_t num](){ 
    uint8_t mask = 1;
    mask <<= num;
    atomic {
      queuedRequests &= ~mask;
      if (owner == num && 
          (cmode == SINGLE_CHANNEL_SINGLE_CONVERSION ||
           cmode == SEQUENCE_OF_CHANNELS ||
           cmode == REPEAT_SINGLE_CHANNEL ||
           cmode == REPEAT_SEQUENCE_OF_CHANNELS)
         )
        cancelled = TRUE;
    }
  }
  
  command uint8_t MSP430ADC12External.getMaxSequenceSize[uint8_t num](){ 
    return 16-maxExtIf; 
  }
  
  command void MSP430ADC12External.setMaxExtInterfaces[uint8_t num](uint8_t max){
    maxExtIf = max;
  }
  
  /**************************** MSP430ADC12Advanced ***********************************/
  
  command result_t MSP430ADC12Advanced.bind[uint8_t num](MSP430ADC12AdvancedSettings_t settings){
    if (num < uniqueCount("MSP430ADC12Advanced"))
      advancedSettings[num] = settings;
    else
     return FAIL;
    return SUCCESS;
  }
    
  command result_t MSP430ADC12Advanced.getSingleDataRepeat[uint8_t num](uint16_t jiffies){
    bool gotIt = FALSE;
       
    if (num >= uniqueCount("MSP430ADC12Advanced"))
      return FAIL;
    atomic {
      if (cmode == ADC_IDLE){
        cmode = ADVANCED_REPEAT_SINGLE_CHANNEL;
        owner = num;
        gotIt = TRUE;
      }
    }
    if (gotIt){
      MSP430ADC12AdvancedSettings_t settings = advancedSettings[num];
      adc12ctl1_t ctl1 = {adc12busy:0, conseq:2, adc12ssel:settings.clockSource, 
                          adc12div:settings.clockDiv, issh:0, shp:1, 
                          shs:settings.sampleHoldSource, cstartadd:8};
      adc12memctl_t memctl = {inch:settings.externalChannel, sref:settings.referenceVoltage, eos:1};
      MSP430CompareControl_t control = {
         ccifg : 0, cov : 0, out : 0, cci : 0, ccie : 1,
         outmod : 1,   // output mode: set (reset in eventhandler from TB isr)
         cap : 0, clld : 0, // loads on write to CCRx
         scs : 0, ccis : 0, cm : 0            
         };
      advancedInterval = jiffies;
      
      // first setup the ADC, waiting for the timer to trigger conversion
      call HPLADC12.disableConversion();
      call HPLADC12.on();
      call HPLADC12.setControl1(ctl1);
      call HPLADC12.setMemControl(8, memctl);
      call HPLADC12.resetMSC();
      call HPLADC12.setSHT(((uint8_t) settings.sampleHoldTime) << 4);
      call HPLADC12.setIEFlags(0x0100);
      call HPLADC12.startConversion();
      
      // now setup and start the timer 
      switch(settings.sampleHoldSource)
      {
        case TIMERA_OUT1: advancedTimer = TIMERA_OUT1; 
                          call CompareA1.setControl(control);
                          call CompareA1.setEventFromNow(1);
                          break;
        case TIMERB_OUT0: advancedTimer = TIMERB_OUT0; 
                          call CompareB0.setControl(control);
                          call CompareB0.setEventFromNow(1);
                          break;
        case TIMERB_OUT1: advancedTimer = TIMERB_OUT1; 
                          call CompareB1.setControl(control);
                          call CompareB1.setEventFromNow(1);
                          break;
      }
    } else 
      return FAIL;
    return SUCCESS;  
  }   
    
  command result_t MSP430ADC12Advanced.getSequenceData[uint8_t num](uint16_t dataDest[], uint8_t length, uint16_t jiffies){
    bool gotIt = FALSE;
    uint8_t maxseqSize = call MSP430ADC12Advanced.getMaxSequenceSize[0]();
    
    if (num >= uniqueCount("MSP430ADC12Advanced") || !length || length > maxseqSize)
      return FAIL;
    atomic {
      if (cmode == ADC_IDLE){
        owner = num;
        cmode = ADVANCED_SEQUENCE_OF_CHANNELS;
        gotIt = TRUE;
      }
    }
    if (gotIt){
      uint8_t i;
      uint16_t mask = 1;
      MSP430ADC12AdvancedSettings_t settings = advancedSettings[num];
      uint8_t myshs=settings.sampleHoldSource;
      adc12ctl1_t ctl1 = {adc12busy:0, conseq:1, adc12ssel:settings.clockSource, 
                          adc12div:settings.clockDiv, issh:0, shp:1, 
                          shs:settings.sampleHoldSource, cstartadd:maxExtIf};
      adc12memctl_t memctl = {inch:settings.externalChannel, sref:settings.referenceVoltage, eos:0};
      MSP430CompareControl_t control = {
         ccifg : 0, cov : 0, out : 0, cci : 0, ccie : 1,
         outmod : 1,   // output mode: set (reset in eventhandler from TB isr)
         cap : 0, clld : 0, // loads on write to CCRx
         scs : 0, ccis : 0, cm : 0            
         };
      advancedInterval = jiffies;
      seqResultPtr = dataDest;
      
      // first setup the ADC, waiting for the timer to trigger conversion
      call HPLADC12.disableConversion();
      for (i=maxExtIf; i<maxExtIf+length-1; i++)
        call HPLADC12.setMemControl(i,memctl);
      memctl.eos = 1;  
      call HPLADC12.setMemControl(i,memctl);
      call HPLADC12.on();
      call HPLADC12.setControl1(ctl1);
      call HPLADC12.resetMSC();
      call HPLADC12.setSHT(((uint8_t) settings.sampleHoldTime) << 4);
      call HPLADC12.setIEFlags(mask << (maxExtIf+length-1));
      call HPLADC12.startConversion();
      
      // now setup and start the timer 
      switch(myshs)
      {
        case TIMERA_OUT1: advancedTimer = TIMERA_OUT1; 
                          call CompareA1.setControl(control);
                          call CompareA1.setEventFromNow(1);
                          break;
        case TIMERB_OUT0: advancedTimer = TIMERB_OUT0; 
                          call CompareB0.setControl(control);
                          call CompareB0.setEventFromNow(1);
                          break;
        case TIMERB_OUT1: advancedTimer = TIMERB_OUT1; 
                          call CompareB1.setControl(control);
                          call CompareB1.setEventFromNow(1);
                          break;
      }
    } else 
      return FAIL;
    return SUCCESS;  
  }
  
  command result_t MSP430ADC12Advanced.getSequenceDataRepeat[uint8_t num](uint16_t dataDest[], uint8_t length, uint16_t jiffies){
    bool gotIt = FALSE;
    uint8_t maxseqSize = call MSP430ADC12Advanced.getMaxSequenceSize[0]();
    
    if (num >= uniqueCount("MSP430ADC12Advanced") || !length || length > maxseqSize)
      return FAIL;
    atomic {
      if (cmode == ADC_IDLE){
        owner = num;
        cmode = ADVANCED_REPEAT_SEQUENCE_OF_CHANNELS;
        gotIt = TRUE;
      }
    }
    if (gotIt){
      uint8_t i;
      uint16_t mask = 1;
      MSP430ADC12AdvancedSettings_t settings = advancedSettings[num];
      adc12ctl1_t ctl1 = {adc12busy:0, conseq:3, adc12ssel:settings.clockSource, 
                          adc12div:settings.clockDiv, issh:0, shp:1, 
                          shs:settings.sampleHoldSource, cstartadd:maxExtIf};
      adc12memctl_t memctl = {inch:settings.externalChannel, sref:settings.referenceVoltage, eos:0};
      MSP430CompareControl_t control = {
         ccifg : 0, cov : 0, out : 0, cci : 0, ccie : 1,
         outmod : 1,   // output mode: set (reset in eventhandler from TB isr)
         cap : 0, clld : 0, // loads on write to CCRx
         scs : 0, ccis : 0, cm : 0            
         };
      advancedInterval = jiffies;
      seqResultPtr = dataDest;
      
      // first setup the ADC, waiting for the timer to trigger conversion
      call HPLADC12.disableConversion();
      for (i=maxExtIf; i<maxExtIf+length-1; i++)
        call HPLADC12.setMemControl(i,memctl);
      memctl.eos = 1;  
      call HPLADC12.setMemControl(i,memctl);
      call HPLADC12.on();
      call HPLADC12.setControl1(ctl1);
      call HPLADC12.resetMSC();
      call HPLADC12.setSHT(((uint8_t) settings.sampleHoldTime) << 4);
      call HPLADC12.setIEFlags(mask << (maxExtIf+length-1));
      call HPLADC12.startConversion();
      
      // now setup and start the timer 
      switch(settings.sampleHoldSource)
      {
        case TIMERA_OUT1: advancedTimer = TIMERA_OUT1; 
                          call CompareA1.setControl(control);
                          call CompareA1.setEventFromNow(1);
                          break;
        case TIMERB_OUT0: advancedTimer = TIMERB_OUT0; 
                          call CompareB0.setControl(control);
                          call CompareB0.setEventFromNow(1);
                          break;
        case TIMERB_OUT1: advancedTimer = TIMERB_OUT1; 
                          call CompareB1.setControl(control);
                          call CompareB1.setEventFromNow(1);
                          break;
      }
    } else 
      return FAIL;
    return SUCCESS;  
  }
  
  async command void MSP430ADC12Advanced.stop[uint8_t num](){
    atomic {
      if (owner == num && 
          (cmode == ADVANCED_SEQUENCE_OF_CHANNELS ||
           cmode == ADVANCED_REPEAT_SINGLE_CHANNEL ||
           cmode == ADVANCED_REPEAT_SEQUENCE_OF_CHANNELS)
         )
        cancelled = TRUE;
    }
  }
  
  command uint8_t MSP430ADC12Advanced.getMaxSequenceSize[uint8_t num](){
    return 16-maxExtIf; 
  }

  async event void CompareA1.fired(){
    MSP430CompareControl_t control1 = {
      ccifg : 0, cov : 0, out : 0, cci : 0, ccie : 1,
      outmod : 0,   // output mode: set
      cap : 0, clld : 0, // loads on write to CCRx
      scs : 0, ccis : 0, cm : 0            
      };
    MSP430CompareControl_t control2 = {
      ccifg : 0, cov : 0, out : 0, cci : 0, ccie : 1,
      outmod : 1,   // output mode: set
      cap : 0, clld : 0, // loads on write to CCRx
      scs : 0, ccis : 0, cm : 0            
      };
    if (advancedTimer ==  TIMERA_OUT1){
      call CompareA1.setControl(control1);  // clear SHI
      call CompareA1.setControl(control2);    
      call CompareA1.setEventFromPrev(advancedInterval); 
    }
  }  
  
  async event void CompareB0.fired(){
    MSP430CompareControl_t control1 = {
      ccifg : 0, cov : 0, out : 0, cci : 0, ccie : 1,
      outmod : 0,   // output mode: set
      cap : 0, clld : 0, // loads on write to CCRx
      scs : 0, ccis : 0, cm : 0            
      };
    MSP430CompareControl_t control2 = {
      ccifg : 0, cov : 0, out : 0, cci : 0, ccie : 1,
      outmod : 1,   // output mode: set
      cap : 0, clld : 0, // loads on write to CCRx
      scs : 0, ccis : 0, cm : 0            
      };
    if (advancedTimer ==  TIMERB_OUT0){
      call CompareB0.setControl(control1);  // clear SHI
      call CompareB0.setControl(control2);    
      call CompareB0.setEventFromPrev(advancedInterval);
    }
  }
  
  async event void CompareB1.fired(){
    MSP430CompareControl_t control1 = {
      ccifg : 0, cov : 0, out : 0, cci : 0, ccie : 1,
      outmod : 0,   // output mode: set
      cap : 0, clld : 0, // loads on write to CCRx
      scs : 0, ccis : 0, cm : 0            
      };
    MSP430CompareControl_t control2 = {
      ccifg : 0, cov : 0, out : 0, cci : 0, ccie : 1,
      outmod : 1,   // output mode: set
      cap : 0, clld : 0, // loads on write to CCRx
      scs : 0, ccis : 0, cm : 0            
      };
    if (advancedTimer == TIMERB_OUT1){
      call CompareB1.setControl(control1);  // clear SHI
      call CompareB1.setControl(control2);
      call CompareB1.setEventFromPrev(advancedInterval);
    }
  }
  
  /**************************** MSP430ADC12Internal ***********************************/
  
  command result_t MSP430ADC12Internal.getData(MSP430ADC12InternalChannel_t channel){ 
    bool gotIt = FALSE;
    atomic {
      if (cmode == ADC_IDLE){
        cmode = INTERNAL_CHANNEL;
        gotIt = TRUE;
      }
    }
    if (gotIt){ // trigger conversion
      uint16_t ctl0 = ADC12CTL0_INTERNAL;
      uint16_t ctl1 = ADC12CTL1_INTERNAL;
      adc12memctl_t extRef = {inch:8, sref:1, eos:1};
      adc12memctl_t refVRatio = {inch:9, sref:1, eos:1};
      adc12memctl_t intTemp = {inch:10, sref:1, eos:1};
      adc12memctl_t battery = {inch:11, sref:1, eos:1};
           
      call HPLADC12.disableConversion();
      call HPLADC12.setControl0_IgnoreRef(*(adc12ctl0_t*) &ctl0);
      call HPLADC12.setControl1(*(adc12ctl1_t*) &ctl1);
      switch(channel){
        case EXTERNAL_REFERENCE: 
          call HPLADC12.setMemControl(8, extRef); break;
        case REFERENCE_VOLTAGE_RATIO: 
          call HPLADC12.setMemControl(8, refVRatio); break;
        case INTERNAL_TEMPERATURE: 
          call HPLADC12.setMemControl(8, intTemp); break;
        case BATTERY_LEVEL: 
          call HPLADC12.setMemControl(8, battery); break;
      }
      call HPLADC12.setIEFlags(0x0100);
      call HPLADC12.startConversion();
      return SUCCESS;
    } 
    return FAIL;
  }
  
  /**************************** internal ***********************************/
  
  default event void MSP430ADC12External.dataReadySingle[uint8_t num](uint16_t x){}
  default event void MSP430ADC12External.dataReadySequence[uint8_t num](){}
  default async event void MSP430ADC12External.dataReadySingleRpt[uint8_t num](uint16_t data){}
  default async event void MSP430ADC12External.dataReadySequenceRpt[uint8_t num](){}
  
  default event void MSP430ADC12Advanced.dataReadySequence[uint8_t num](){}
  default async event void MSP430ADC12Advanced.dataReadySingleRpt[uint8_t num](uint16_t data){}
  default async event void MSP430ADC12Advanced.dataReadySequenceRpt[uint8_t num](){}
  
  default event void MSP430ADC12Internal.dataReady(MSP430ADC12InternalChannel_t channel, uint16_t data){}  
  
  task void conversionDone()  // single-(not repeat)-mode postprocessing
  {
    uint16_t data = 0; 
    uint8_t i, mask = 1;
    MSP430ADC12InternalChannel_t channel = 0;
    adc12memctl_t memctl;
        
    switch (cmode){
      case SEQUENCE_OF_CHANNELS:
          queuedRequests &=  ~(mask << owner); 
      case ADVANCED_SEQUENCE_OF_CHANNELS:
          i = maxExtIf;
          do {
            memctl = call HPLADC12.getMemControl(i);
            *seqResultPtr++ = call HPLADC12.getMem(i++);
          } while (!memctl.eos);
          cmode = ADC_IDLE;  // enable access to ADC
          if (cmode == SEQUENCE_OF_CHANNELS)
            signal MSP430ADC12External.dataReadySequence[owner](); 
          else {
            uint16_t ctl0 = ADC12CTL0_DEFAULT;
            uint16_t ctl1 = ADC12CTL1_DEFAULT;
            call HPLADC12.disableConversion();
            call HPLADC12.setControl0_IgnoreRef(*(adc12ctl0_t*) &ctl0);
            call HPLADC12.setControl1(*(adc12ctl1_t*) &ctl1);
            signal MSP430ADC12Advanced.dataReadySequence[owner](); 
          }
          break;
      case SINGLE_CHANNEL_SINGLE_CONVERSION:
          queuedRequests &=  ~(mask << owner); 
          data = call HPLADC12.getMem(owner);
          cmode = ADC_IDLE;  // enable access to ADC
          signal MSP430ADC12External.dataReadySingle[owner](data); 
          break;
      case INTERNAL_CHANNEL:{
          uint16_t ctl0 = ADC12CTL0_DEFAULT;
          uint16_t ctl1 = ADC12CTL1_DEFAULT;
          call HPLADC12.disableConversion();
          call HPLADC12.setControl0_IgnoreRef(*(adc12ctl0_t*) &ctl0);
          call HPLADC12.setControl1(*(adc12ctl1_t*) &ctl1);
          data = call HPLADC12.getMem(8);
          memctl = call HPLADC12.getMemControl(8);
          channel = memctl.inch;
          cmode = ADC_IDLE;  // enable access to ADC
          signal MSP430ADC12Internal.dataReady(channel, data);
          }
          break;
      default: break;
    }
    
    post checkQueue();
  }

  task void checkQueue()
  {
    uint8_t i, isEntry = 0, mask = 1;
    mask <<= queueOffset;
    
    if (cmode != ADC_IDLE || !queuedRequests)
      return; // nothing to do
    
    for (i=0; i<8; i++){ // find next one
      if (queuedRequests & mask)
        break;
      mask <<= 1;
      if (!mask)
        mask = 1;
    }      
    atomic {    
      if (cmode == ADC_IDLE && (queuedRequests & mask)){ // need to be sure (concurrency)
        queuedRequests &=  ~mask; 
        queueOffset = (queueOffset+i) % 8;    
        cmode = SINGLE_CHANNEL_SINGLE_CONVERSION;
        owner = queueOffset;
        queueOffset = (queueOffset+1) % 8; ;  // to next entry
        isEntry = 1;
      } else {
        // bad luck, someone stopped request in between
        post checkQueue();
      }
    }
    if (isEntry){
      call HPLADC12.disableConversion();
      call HPLADC12.on();
      call HPLADC12.setConversionMode(SINGLE_CHANNEL_SINGLE_CONVERSION, queueOffset-1);
      call HPLADC12.setSHT((uint8_t) (samplingTime >> (queueOffset-1) * 4));
      call HPLADC12.setIEFlags(mask);
      call HPLADC12.startConversion(); 
    }
  }    
      
  void disableTimers(){
    if (cmode == ADVANCED_SEQUENCE_OF_CHANNELS ||
        cmode == ADVANCED_REPEAT_SINGLE_CHANNEL ||
        cmode == ADVANCED_REPEAT_SEQUENCE_OF_CHANNELS)
    {
      switch(advancedTimer)
      {
        case TIMERA_OUT1: call CompareA1.disableEvents(); break;
        case TIMERB_OUT0: call CompareB0.disableEvents(); break;
        case TIMERB_OUT1: call CompareB1.disableEvents(); break;
      }  
    }
  }  
   
  async event void HPLADC12.converted(uint8_t number){
    // first clear IFG (for compiler option -Os data needs to be volatile)
    volatile uint16_t data = call HPLADC12.getMem(number);  
    
    if (cancelled == TRUE) {
      disableTimers();  
      call HPLADC12.stopConversion();
      call HPLADC12.resetIFGs();
      cmode = ADC_IDLE;
      cancelled = FALSE;
      post checkQueue();
    
    } else {   // not cancelled 
      
      switch (cmode) // what was it ?
      {
        case REPEAT_SINGLE_CHANNEL:
            signal MSP430ADC12External.dataReadySingleRpt[owner](data);
            break;
        case REPEAT_SEQUENCE_OF_CHANNELS: 
            {
              uint8_t i = maxExtIf;
              adc12memctl_t memctl;
              do {
                memctl = call HPLADC12.getMemControl(i);
                *seqResultPtr++ = call HPLADC12.getMem(i++);
              } while (!memctl.eos);
              seqResultPtr -= i-maxExtIf;
              signal MSP430ADC12External.dataReadySequenceRpt[owner]();  
            }
            break;
        case ADVANCED_REPEAT_SINGLE_CHANNEL:
            signal MSP430ADC12Advanced.dataReadySingleRpt[owner](data);
            break;
        case ADVANCED_REPEAT_SEQUENCE_OF_CHANNELS:
            {
              uint8_t i = maxExtIf;
              adc12memctl_t memctl;
              do {
                memctl = call HPLADC12.getMemControl(i);
                *seqResultPtr++ = call HPLADC12.getMem(i++);
              } while (!memctl.eos);
              seqResultPtr -= i-maxExtIf;
              signal MSP430ADC12Advanced.dataReadySequenceRpt[owner](); 
            }
            break;
        case ADVANCED_SEQUENCE_OF_CHANNELS:
            disableTimers();  
            call HPLADC12.stopConversion();
            call HPLADC12.resetIFGs();
            // go on with default
        default:
            // SINGLE_CHANNEL_SINGLE_CONVERSION or SEQUENCE_OF_CHANNELS or 
            // INTERNAL_CHANNEL or ADVANCED_SEQUENCE_OF_CHANNELS 
            post conversionDone();
            break;
      } // switch
    } // else 
  }
  
  async event void HPLADC12.memOverflow(){}
  async event void HPLADC12.timeOverflow(){}

}

