//$Id$
//@author Cory Sharp <cssharp@eecs.berkeley.edu>

includes MSP430Timer;

module MSP430TimerM
{
  provides interface MSP430Timer as TimerA;
  provides interface MSP430Compare as CompareA0;
  provides interface MSP430Compare as CompareA1;
  provides interface MSP430Compare as CompareA2;

  provides interface MSP430Timer as TimerB;
  provides interface MSP430Compare as CompareB0;
  provides interface MSP430Compare as CompareB1;
  provides interface MSP430Compare as CompareB2;
  provides interface MSP430Compare as CompareB3;
  provides interface MSP430Compare as CompareB4;
  provides interface MSP430Compare as CompareB5;
  provides interface MSP430Compare as CompareB6;
}
implementation
{
  MSP430REG_NORACE(TACCTL0);
  MSP430REG_NORACE(TACCTL1);
  MSP430REG_NORACE(TACCTL2);

  MSP430REG_NORACE(TACCR0);
  MSP430REG_NORACE(TACCR1);
  MSP430REG_NORACE(TACCR2);

  MSP430REG_NORACE(TBCCTL0);
  MSP430REG_NORACE(TBCCTL1);
  MSP430REG_NORACE(TBCCTL2);
  MSP430REG_NORACE(TBCCTL3);
  MSP430REG_NORACE(TBCCTL4);
  MSP430REG_NORACE(TBCCTL5);
  MSP430REG_NORACE(TBCCTL6);

  MSP430REG_NORACE(TBCCR0);
  MSP430REG_NORACE(TBCCR1);
  MSP430REG_NORACE(TBCCR2);
  MSP430REG_NORACE(TBCCR3);
  MSP430REG_NORACE(TBCCR4);
  MSP430REG_NORACE(TBCCR5);
  MSP430REG_NORACE(TBCCR6);

  typedef MSP430CompareControl_t CC_t;

  uint16_t timerControl()
  {
    CC_t x = {
      cm : 1,    // capture on rising edge
      ccis : 0,  // capture/compare input select
      clld : 0,  // TBCL1 loads on write to TBCCR1
      cap : 0,   // compare mode
      ccie : 0,  // capture compare interrupt enable 
    };
    return *(uint16_t*)&x;
  }

  TOSH_SIGNAL(TIMERA0_VECTOR)
  {
    signal CompareA0.fired();
  }

  TOSH_SIGNAL(TIMERA1_VECTOR)
  {
    int n = TAIV;
    switch( n )
    {
      case  0: break;
      case  2: signal CompareA1.fired(); break;
      case  4: signal CompareA2.fired(); break;
      case  6: break;
      case  8: break;
      case 10: signal TimerA.overflow(); break;
      case 12: break;
      case 14: break;
    }
  }

  default async event void CompareA0.fired() { }
  default async event void CompareA1.fired() { }
  default async event void CompareA2.fired() { }
  default async event void TimerA.overflow() { }

  async command uint16_t TimerA.read() { return TAR; }
  async command uint16_t TimerB.read() { return TBR; }

  async command bool TimerA.isOverflowPending() { return TACTL & TAIFG; }
  async command bool TimerB.isOverflowPending() { return TBCTL & TBIFG; }

  async command void TimerA.clearOverflow() { CLR_FLAG(TACTL,TAIFG); }
  async command void TimerB.clearOverflow() { CLR_FLAG(TBCTL,TBIFG); }

  async command CC_t CompareA0.getControl() { return *(CC_t*)&TACCTL0; }
  async command CC_t CompareA1.getControl() { return *(CC_t*)&TACCTL1; }
  async command CC_t CompareA2.getControl() { return *(CC_t*)&TACCTL2; }

  async command bool CompareA0.isInterruptPending() { return TACCTL0 & CCIFG; }
  async command bool CompareA1.isInterruptPending() { return TACCTL1 & CCIFG; }
  async command bool CompareA2.isInterruptPending() { return TACCTL2 & CCIFG; }

  async command void CompareA0.clearPendingInterrupt() { CLR_FLAG(TACCTL0,CCIFG); }
  async command void CompareA1.clearPendingInterrupt() { CLR_FLAG(TACCTL1,CCIFG); }
  async command void CompareA2.clearPendingInterrupt() { CLR_FLAG(TACCTL2,CCIFG); }

  async command void CompareA0.setControl( CC_t x ) { TACCTL0 = *(uint16_t*)&x; }
  async command void CompareA1.setControl( CC_t x ) { TACCTL1 = *(uint16_t*)&x; }
  async command void CompareA2.setControl( CC_t x ) { TACCTL2 = *(uint16_t*)&x; }

  async command void CompareA0.setControlAsTimer() { TACCTL0 = timerControl(); }
  async command void CompareA1.setControlAsTimer() { TACCTL1 = timerControl(); }
  async command void CompareA2.setControlAsTimer() { TACCTL2 = timerControl(); }

  async command void CompareA0.enableEvents() { SET_FLAG( TACCTL0, CCIE ); }
  async command void CompareA1.enableEvents() { SET_FLAG( TACCTL1, CCIE ); }
  async command void CompareA2.enableEvents() { SET_FLAG( TACCTL2, CCIE ); }

  async command void CompareA0.disableEvents() { CLR_FLAG( TACCTL0, CCIE ); }
  async command void CompareA1.disableEvents() { CLR_FLAG( TACCTL1, CCIE ); }
  async command void CompareA2.disableEvents() { CLR_FLAG( TACCTL2, CCIE ); }

  async command uint16_t CompareA0.getEvent() { return TACCR0; }
  async command uint16_t CompareA1.getEvent() { return TACCR1; }
  async command uint16_t CompareA2.getEvent() { return TACCR2; }

  async command void CompareA0.setEvent( uint16_t x ) { TACCR0 = x; }
  async command void CompareA1.setEvent( uint16_t x ) { TACCR1 = x; }
  async command void CompareA2.setEvent( uint16_t x ) { TACCR2 = x; }

  async command void CompareA0.setEventFromPrev( uint16_t x ) { TACCR0 += x; }
  async command void CompareA1.setEventFromPrev( uint16_t x ) { TACCR1 += x; }
  async command void CompareA2.setEventFromPrev( uint16_t x ) { TACCR2 += x; }

  async command void CompareA0.setEventFromNow( uint16_t x ) { TACCR0 = TAR+x; }
  async command void CompareA1.setEventFromNow( uint16_t x ) { TACCR1 = TAR+x; }
  async command void CompareA2.setEventFromNow( uint16_t x ) { TACCR2 = TAR+x; }


  TOSH_SIGNAL(TIMERB0_VECTOR)
  {
    signal CompareB0.fired();
  }

  TOSH_SIGNAL(TIMERB1_VECTOR)
  {
    int n = TBIV;
    switch( n )
    {
      case  0: break;
      case  2: signal CompareB1.fired(); break;
      case  4: signal CompareB2.fired(); break;
      case  6: signal CompareB3.fired(); break;
      case  8: signal CompareB4.fired(); break;
      case 10: signal CompareB5.fired(); break;
      case 12: signal CompareB6.fired(); break;
      case 14: signal TimerB.overflow(); break;
    }
  }

  default async event void CompareB0.fired() { }

  default async event void CompareB1.fired() { }
  default async event void CompareB2.fired() { }
  default async event void CompareB3.fired() { }
  default async event void CompareB4.fired() { }
  default async event void CompareB5.fired() { }
  default async event void CompareB6.fired() { }
  default async event void TimerB.overflow() { }

  async command CC_t CompareB0.getControl() { return *(CC_t*)&TBCCTL0; }
  async command CC_t CompareB1.getControl() { return *(CC_t*)&TBCCTL1; }
  async command CC_t CompareB2.getControl() { return *(CC_t*)&TBCCTL2; }
  async command CC_t CompareB3.getControl() { return *(CC_t*)&TBCCTL3; }
  async command CC_t CompareB4.getControl() { return *(CC_t*)&TBCCTL4; }
  async command CC_t CompareB5.getControl() { return *(CC_t*)&TBCCTL5; }
  async command CC_t CompareB6.getControl() { return *(CC_t*)&TBCCTL6; }

  async command bool CompareB0.isInterruptPending() { return TBCCTL0 & CCIFG; }
  async command bool CompareB1.isInterruptPending() { return TBCCTL1 & CCIFG; }
  async command bool CompareB2.isInterruptPending() { return TBCCTL2 & CCIFG; }
  async command bool CompareB3.isInterruptPending() { return TBCCTL3 & CCIFG; }
  async command bool CompareB4.isInterruptPending() { return TBCCTL4 & CCIFG; }
  async command bool CompareB5.isInterruptPending() { return TBCCTL5 & CCIFG; }
  async command bool CompareB6.isInterruptPending() { return TBCCTL6 & CCIFG; }

  async command void CompareB0.clearPendingInterrupt() { CLR_FLAG(TBCCTL0,CCIFG); }
  async command void CompareB1.clearPendingInterrupt() { CLR_FLAG(TBCCTL1,CCIFG); }
  async command void CompareB2.clearPendingInterrupt() { CLR_FLAG(TBCCTL2,CCIFG); }
  async command void CompareB3.clearPendingInterrupt() { CLR_FLAG(TBCCTL3,CCIFG); }
  async command void CompareB4.clearPendingInterrupt() { CLR_FLAG(TBCCTL4,CCIFG); }
  async command void CompareB5.clearPendingInterrupt() { CLR_FLAG(TBCCTL5,CCIFG); }
  async command void CompareB6.clearPendingInterrupt() { CLR_FLAG(TBCCTL6,CCIFG); }

  async command void CompareB0.setControl( CC_t x ) { TBCCTL0 = *(uint16_t*)&x; }
  async command void CompareB1.setControl( CC_t x ) { TBCCTL1 = *(uint16_t*)&x; }
  async command void CompareB2.setControl( CC_t x ) { TBCCTL2 = *(uint16_t*)&x; }
  async command void CompareB3.setControl( CC_t x ) { TBCCTL3 = *(uint16_t*)&x; }
  async command void CompareB4.setControl( CC_t x ) { TBCCTL4 = *(uint16_t*)&x; }
  async command void CompareB5.setControl( CC_t x ) { TBCCTL5 = *(uint16_t*)&x; }
  async command void CompareB6.setControl( CC_t x ) { TBCCTL6 = *(uint16_t*)&x; }

  async command void CompareB0.setControlAsTimer() { TBCCTL0 = timerControl(); }
  async command void CompareB1.setControlAsTimer() { TBCCTL1 = timerControl(); }
  async command void CompareB2.setControlAsTimer() { TBCCTL2 = timerControl(); }
  async command void CompareB3.setControlAsTimer() { TBCCTL3 = timerControl(); }
  async command void CompareB4.setControlAsTimer() { TBCCTL4 = timerControl(); }
  async command void CompareB5.setControlAsTimer() { TBCCTL5 = timerControl(); }
  async command void CompareB6.setControlAsTimer() { TBCCTL6 = timerControl(); }

  async command void CompareB0.enableEvents() { SET_FLAG( TBCCTL0, CCIE ); }
  async command void CompareB1.enableEvents() { SET_FLAG( TBCCTL1, CCIE ); }
  async command void CompareB2.enableEvents() { SET_FLAG( TBCCTL2, CCIE ); }
  async command void CompareB3.enableEvents() { SET_FLAG( TBCCTL3, CCIE ); }
  async command void CompareB4.enableEvents() { SET_FLAG( TBCCTL4, CCIE ); }
  async command void CompareB5.enableEvents() { SET_FLAG( TBCCTL5, CCIE ); }
  async command void CompareB6.enableEvents() { SET_FLAG( TBCCTL6, CCIE ); }

  async command void CompareB0.disableEvents() { CLR_FLAG( TBCCTL0, CCIE ); }
  async command void CompareB1.disableEvents() { CLR_FLAG( TBCCTL1, CCIE ); }
  async command void CompareB2.disableEvents() { CLR_FLAG( TBCCTL2, CCIE ); }
  async command void CompareB3.disableEvents() { CLR_FLAG( TBCCTL3, CCIE ); }
  async command void CompareB4.disableEvents() { CLR_FLAG( TBCCTL4, CCIE ); }
  async command void CompareB5.disableEvents() { CLR_FLAG( TBCCTL5, CCIE ); }
  async command void CompareB6.disableEvents() { CLR_FLAG( TBCCTL6, CCIE ); }

  async command uint16_t CompareB0.getEvent() { return TBCCR0; }
  async command uint16_t CompareB1.getEvent() { return TBCCR1; }
  async command uint16_t CompareB2.getEvent() { return TBCCR2; }
  async command uint16_t CompareB3.getEvent() { return TBCCR3; }
  async command uint16_t CompareB4.getEvent() { return TBCCR4; }
  async command uint16_t CompareB5.getEvent() { return TBCCR5; }
  async command uint16_t CompareB6.getEvent() { return TBCCR6; }

  async command void CompareB0.setEvent( uint16_t x ) { TBCCR0 = x; }
  async command void CompareB1.setEvent( uint16_t x ) { TBCCR1 = x; }
  async command void CompareB2.setEvent( uint16_t x ) { TBCCR2 = x; }
  async command void CompareB3.setEvent( uint16_t x ) { TBCCR3 = x; }
  async command void CompareB4.setEvent( uint16_t x ) { TBCCR4 = x; }
  async command void CompareB5.setEvent( uint16_t x ) { TBCCR5 = x; }
  async command void CompareB6.setEvent( uint16_t x ) { TBCCR6 = x; }

  async command void CompareB0.setEventFromPrev( uint16_t x ) { TBCCR0 += x; }
  async command void CompareB1.setEventFromPrev( uint16_t x ) { TBCCR1 += x; }
  async command void CompareB2.setEventFromPrev( uint16_t x ) { TBCCR2 += x; }
  async command void CompareB3.setEventFromPrev( uint16_t x ) { TBCCR3 += x; }
  async command void CompareB4.setEventFromPrev( uint16_t x ) { TBCCR4 += x; }
  async command void CompareB5.setEventFromPrev( uint16_t x ) { TBCCR5 += x; }
  async command void CompareB6.setEventFromPrev( uint16_t x ) { TBCCR6 += x; }

  async command void CompareB0.setEventFromNow( uint16_t x ) { TBCCR0 = TBR+x; }
  async command void CompareB1.setEventFromNow( uint16_t x ) { TBCCR1 = TBR+x; }
  async command void CompareB2.setEventFromNow( uint16_t x ) { TBCCR2 = TBR+x; }
  async command void CompareB3.setEventFromNow( uint16_t x ) { TBCCR3 = TBR+x; }
  async command void CompareB4.setEventFromNow( uint16_t x ) { TBCCR4 = TBR+x; }
  async command void CompareB5.setEventFromNow( uint16_t x ) { TBCCR5 = TBR+x; }
  async command void CompareB6.setEventFromNow( uint16_t x ) { TBCCR6 = TBR+x; }
}

