/*
 * Copyright (c) 2004, Technische Universität Berlin
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright 
 *   notice, this list of conditions and the following disclaimer in the 
 *   documentation and/or other materials provided with the distribution.
 * - Neither the name of the Technische Universität Berlin nor the names 
 *   of its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED 
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, 
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE 
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * - Description ----------------------------------------------------------
 * This component manages the ADC12's reference voltage generator.
 * The internal turn-on time is 17ms, thus the component is programmed  
 * split-phase, i.e. after the command <code>acquire</code> has been called 
 * you will eventually get the event <code>stable</code> when vref is 
 * stable.
 * The reference should be turned off to save power when not in use
 * with the command <code>release</code>.
 * - Revision -------------------------------------------------------------
 * $Revision$
 * $Date$
 * @author: Jan Hauer (hauer@tkn.tu-berlin.de)
 * ========================================================================
 */
module RefVoltM
{
  provides interface RefVolt;
  uses {
    interface HPLADC12;
    interface TimerMilli;
  }
}

implementation
{
  uint8_t semaCount = 0; // number of interface instaces with successful acquire-calls
  RefVoltState_t state = INTERNAL_REFERENCE_OFF;  // current state of generator
  
  command result_t RefVolt.acquire(RefVolt_t vref)
  {
    bool weSwitchedItOn = FALSE;
    if (semaCount > 0){ // REFON is already set
      if ((call HPLADC12.getRef2_5V() && vref == REFERENCE_2_5V) ||
          (!call HPLADC12.getRef2_5V() && vref == REFERENCE_1_5V)){          
        semaCount++;
        return SUCCESS;
      } else
        return FAIL;
    } else { // REFON is reset
      atomic {
        if (!call HPLADC12.isBusy()){
          call HPLADC12.disableConversion();
          call HPLADC12.setRefOn();
          if (vref == REFERENCE_1_5V)
            call HPLADC12.setRef1_5V();
          else
            call HPLADC12.setRef2_5V();
          weSwitchedItOn = TRUE;
          state = INTERNAL_REFERENCE_UNSTABLE;
        }
      }
      if (weSwitchedItOn){
        semaCount++;
        call TimerMilli.stop();
        call TimerMilli.setOneShot(17);    // 17 ms to become stable
        return SUCCESS;
      }
      return FAIL;
    }
  }
      
  command result_t RefVolt.release()
  {
    bool weSwitchedItOff = FALSE;
    if (semaCount == 0) 
      return FAIL;
    if (--semaCount == 0){
      atomic {
        if (!call HPLADC12.isBusy()){
          call HPLADC12.disableConversion();
          call HPLADC12.setRefOff();
          weSwitchedItOff = TRUE;
          state = INTERNAL_REFERENCE_OFF;
        }
      }
      if (!weSwitchedItOff){
        // damn, we cant switch it off now! 
        call TimerMilli.stop();
        state = INTERNAL_REFERENCE_KILLED;
        call TimerMilli.setOneShot(SWITCH_OFF_TRY_MS);
      }
    }
    return SUCCESS;
  }
  
  command RefVolt_t RefVolt.getState()
  {
    if (state == INTERNAL_REFERENCE_STABLE)
      if (call HPLADC12.getRef2_5V())
        return REFERENCE_2_5V;
      else
        return REFERENCE_1_5V;
    return REFERENCE_UNSTABLE;
  }
  
  event result_t TimerMilli.fired()
  {
    bool weSwitchedItOff = FALSE;
    switch (state){
      case INTERNAL_REFERENCE_UNSTABLE:
          state = INTERNAL_REFERENCE_STABLE;
          if (call HPLADC12.getRef2_5V())
            signal RefVolt.stateChanged(REFERENCE_2_5V);
          else 
            signal RefVolt.stateChanged(REFERENCE_1_5V);
          break;
      case INTERNAL_REFERENCE_KILLED:
          atomic {
            if (!call HPLADC12.isBusy()){
              call HPLADC12.disableConversion();
              call HPLADC12.setRefOff();
              weSwitchedItOff = TRUE;
              state = INTERNAL_REFERENCE_OFF;
            }
          }
          if (!weSwitchedItOff){
            // damn, we cant switch it off now! 
            call TimerMilli.setOneShot(SWITCH_OFF_TRY_MS);
          }
          break;
      default:
    }
    return FALSE;
  }

  async event void HPLADC12.memOverflow(){}
  async event void HPLADC12.timeOverflow(){}
  async event void HPLADC12.converted(uint8_t number){}
}



