/**
 * Copyright (C) 1998-2000 by University of Maryland, College Park, MD 20742, USA
 * All rights reserved.
 */
package edu.umd.cs.jazz.event;

import java.awt.AWTEvent;
import java.awt.geom.AffineTransform;
import java.io.*;

import edu.umd.cs.jazz.*;

/**
 * <b>ZGroupEvent</b> is an event which indicates that a group node has changed.
 * <P>
 * Group events are provided for notification purposes ONLY;
 * Jazz will automatically handle changes to the group
 * contents internally so that the program works properly regardless of
 * whether the program is receiving these events or not.
 * <P>
 * This event is generated by a ZGroup
 * when a node is added or removed from it.
 * The event is passed to every <code>ZGroupListener</code>
 * or <code>ZGroupAdapter</code> object which registered to receive such
 * events using the group's <code>addGroupListener</code> method.
 * (<code>ZGroupAdapter</code> objects implement the
 * <code>ZGroupListener</code> interface.) Each such listener object
 * gets this <code>ZGroupEvent</code> when the event occurs.
 * <P>
 * ZGroupEvents now contains a method <code>isModificationEvent()</code> to
 * distinguish a modification event from a <bold>true</bold> node addition or
 * removal.  A modification event is one in which a node changes position
 * in a single scenegraph or between two different scenegraphs.  A true
 * addition or removal event is one in which a node is first added to or
 * removed from a scenegraph.  For example, the following bit of code checks
 * to see if a given node has been deleted from its parent by checking to
 * see if the event is a modification event.
 * <pre>
 *     public void nodeRemoved(ZGroupEvent e) {
 *         if (e.getChild() == myNode &&
 *             !e.isModificationEvent()) {
 *             // myNode was *truly* deleted from the scenegraph
 *         }
 *     }
 * </pre>
 *
 * <P>
 * <b>Warning:</b> Serialized and ZSerialized objects of this class will not be
 * compatible with future Jazz releases. The current serialization support is
 * appropriate for short term storage or RMI between applications running the
 * same version of Jazz. A future release of Jazz will provide support for long
 * term persistence.
 *
 * @see ZGroupAdapter
 * @see ZGroupListener
 * @author Ben Bederson
 */
public class ZGroupEvent extends AWTEvent implements ZEvent, Serializable {

    /**
     * The first number in the range of ids used for group events.
     */
    public static final int GROUP_FIRST       = 100;

    /**
     * The last number in the range of ids used for group events.
     */
    public static final int GROUP_LAST        = 101;

    /**
     * This event indicates that a node was added to the group.
     */
    public static final int NODE_ADDED  = GROUP_FIRST;

    /**
     * This event indicates that a node was removed from the group.
     */
    public static final int NODE_REMOVED = 1 + GROUP_FIRST;

    /**
     * The non-null node that is being added or
     * removed from the group.
     *
     * @see #getChild
     */
    ZNode child;

    /**
     * True if this event is a modification.
     */
    private boolean modification;

    /**
     * True if this event has been consumed.
     */
    private boolean consumed;

    /**
     * Constructs a ZGroupEvent object.
     *
     * @param source        the ZGroup object that originated the event
     * @param id            an integer indicating the type of event
     * @param child         the node that was added or removed
     * @param modification  is this event a modification?
     * @deprecated as of Jazz 1.1, use createNodeAddedEvent() or createNodeRemovedEvent() instead.
     */
    public ZGroupEvent(ZGroup source, int id, ZNode child, boolean modification) {
        super(source, id);
        this.child = child;
        this.modification = modification;
        consumed = false;
    }

    protected ZGroupEvent(ZGroup source, int id, ZNode child, boolean modification, Object dummy) {
        super(source, id);
        this.child = child;
        this.modification = modification;
        consumed = false;
    }

    /**
     * Factory method to create a ZGroupEvent with a NODE_ADDED ID.
     *
     * @param source        the ZGroup object that originated the event
     * @param child         the node that was added or removed
     * @param modification  is this event a modification?
     */
    public static ZGroupEvent createNodeAddedEvent(ZGroup source, ZNode child, boolean modification) {
        return new ZGroupEvent(source, ZGroupEvent.NODE_ADDED, child, modification, null);
    }

    /**
     * Factory method to create a ZGroupEvent with a NODE_REMOVED ID.
     *
     * @param source        the ZGroup object that originated the event
     * @param child         the node that was added or removed
     * @param modification  is this event a modification?
     */
    public static ZGroupEvent createNodeRemovedEvent(ZGroup source, ZNode child, boolean modification) {
        return new ZGroupEvent(source, ZGroupEvent.NODE_REMOVED, child, modification, null);
    }

    /**
     * Returns the originator of the event.
     *
     * @return the ZGroup object that originated the event.
     */
    public ZGroup getGroup() {
        return (ZGroup)source;   // Cast is ok, checked in constructor
    }

    /**
     * Returns the ZNode that was affected by the event.
     *
     * @return the ZNode object that was added or removed
     */
    public ZNode getChild() {
        return child;
    }

    /**
     * True if this event is a modification. A modification event is one
     * in which a node changes position in a single scenegraph or between
     * two different scenegraphs.  A non-modification event is one in
     * which a node is first added to, or removed from, a scenegraph.
     *
     * @return Does this event represent a modification?
     */
    public boolean isModificationEvent() {
        return modification;
    }

    /**
     * True if this event has been consumed.
     */
    public boolean isConsumed() {
        return consumed;
    }

    /**
     * Consume this event.
     */
    public void consume() {
        consumed = true;
    }

    /**
     * Calls appropriate method on the listener based on this events ID.
     */
    public void dispatchTo(Object listener) {
        ZGroupListener groupListener = (ZGroupListener) listener;
        switch (getID()) {
            case ZGroupEvent.NODE_ADDED:
                groupListener.nodeAdded(this);
                break;
            case ZGroupEvent.NODE_REMOVED:
                groupListener.nodeRemoved(this);
                break;
            default:
                throw new RuntimeException("ZGroupEvent with bad ID");
        }
    }

    /**
     * Returns the ZGroupLister class.
     */
    public Class getListenerType() {
        return ZGroupListener.class;
    }

    /**
     * Set the souce of this event. As the event is fired up the tree the source of the
     * event will keep changing to reflect the scenegraph object that is firing the event.
     */
    public void setSource(Object aSource) {
        source = aSource;
    }
}