/**
 * Copyright (C) 1998-2000 by University of Maryland, College Park, MD 20742, USA
 * All rights reserved.
 */
package edu.umd.cs.jazz.event;

import java.awt.AWTEvent;
import java.awt.geom.AffineTransform;
import java.io.*;

import edu.umd.cs.jazz.*;

/**
 * <b>ZNodeEvent</b> is an event which indicates that a node has changed.
 * <P>
 * Node events are provided for notification purposes ONLY;
 * Jazz will automatically handle changes to the node
 * contents internally so that the program works properly regardless of
 * whether the program is receiving these events or not.
 * <P>
 * This event is generated by a ZNode when a node's bounds are changed.
 * The event is passed to every <code>ZNodeListener</code>
 * or <code>ZNodeAdapter</code> object which registered to receive such
 * events using the Node's <code>addNodeListener</code> method.
 * (<code>ZNodeAdapter</code> objects implement the
 * <code>ZNodeListener</code> interface.) Each such listener object
 * gets this <code>ZNodeEvent</code> when the event occurs.
 *
 * <P>
 * <b>Warning:</b> Serialized and ZSerialized objects of this class will not be
 * compatible with future Jazz releases. The current serialization support is
 * appropriate for short term storage or RMI between applications running the
 * same version of Jazz. A future release of Jazz will provide support for long
 * term persistence.
 *
 * @see ZNodeAdapter
 * @see ZNodeListener
 * @author Ben Bederson
 */
public class ZNodeEvent extends AWTEvent implements ZEvent, Serializable {

    /**
     * The first number in the range of ids used for node events.
     */
    public static final int NODE_FIRST       = 500;

    /**
     * The last number in the range of ids used for node events.
     */
    public static final int NODE_LAST        = 501;

    /**
     * This event indicates that a node's bounds changed.
     */
    public static final int BOUNDS_CHANGED      = NODE_FIRST;

    /**
     * This event indicates that a node's global bounds changed.
     */
    public static final int GLOBAL_BOUNDS_CHANGED = NODE_LAST;

    /**
     * The non-null node whose bounds changed.
     *
     * @see #getNode
     */
    ZNode node;

    /**
     * True if this event has been consumed;
     */
    private boolean consumed;

    /**
     * Constructs a ZNodeEvent object.
     *
     * @param source    the ZNode object that originated the event, and whose
     * bounds have changed.
     * @param id an integer indicating the type of event
     * @deprecated as of Jazz 1.1, use createBoundsChangedEvent() or createGlobalBoundsChanged() instead.
     */
    public ZNodeEvent(ZNode source, int id) {
        super(source, id);
        node = source;
        consumed = false;
    }

    protected ZNodeEvent(ZNode source, int id, Object dummy) {
        super(source, id);
        node = source;
        consumed = false;
    }

    /**
     * Factory method to create a ZNodeEvent with a BOUNDS_CHANGED ID.
     *
     * @param source The ZNode object that originated the event, and whose
     * bounds have changed.
     */
    public static ZNodeEvent createBoundsChangedEvent(ZNode aNode) {
        return new ZNodeEvent(aNode, ZNodeEvent.BOUNDS_CHANGED, null);
    }

    /**
     * Factory method to create a ZNodeEvent with a GLOBAL_BOUNDS_CHANGED ID.
     *
     * @param source The ZNode object that originated the event, and whose
     * bounds have changed.
     */
    public static ZNodeEvent createGlobalBoundsChangedEvent(ZNode aNode) {
        return new ZNodeEvent(aNode, ZNodeEvent.GLOBAL_BOUNDS_CHANGED, null);
    }

    /**
     * Returns the node whose bounds has changed.
     *
     * @return the ZNode whose bounds has changed.
     */
    public ZNode getNode() {
        return node;
    }

    /**
     * True if this event has been consumed.
     */
    public boolean isConsumed() {
        return consumed;
    }

    /**
     * Consume this event.
     */
    public void consume() {
        consumed = true;
    }

    /**
     * Calls appropriate method on the listener based on this events ID.
     */
    public void dispatchTo(Object listener) {
        ZNodeListener nodeListener = (ZNodeListener) listener;
        switch (getID()) {
            case ZNodeEvent.BOUNDS_CHANGED:
                nodeListener.boundsChanged(this);
                break;
            case ZNodeEvent.GLOBAL_BOUNDS_CHANGED:
                nodeListener.globalBoundsChanged(this);
                break;
            default:
                throw new RuntimeException("ZNodeEvent with bad ID");
        }
    }

    /**
     * Returns the ZNodeLister class.
     */
    public Class getListenerType() {
        return ZNodeListener.class;
    }

    /**
     * Set the souce of this event. As the event is fired up the tree the source of the
     * event will keep changing to reflect the scenegraph object that is firing the event.
     */
    public void setSource(Object aSource) {
        source = aSource;
    }
}