/**
 * Copyright (C) 1998-2000 by University of Maryland, College Park, MD 20742, USA
 * All rights reserved.
 */
package edu.umd.cs.jazz.event;

import java.awt.AWTEvent;
import java.awt.geom.AffineTransform;
import java.io.*;

import edu.umd.cs.jazz.*;

/**
 * <b>ZTransformEvent</b> is an event which indicates that a transform group has changed.
 * <P>
 * Transform events are provided for notification purposes ONLY;
 * Jazz will automatically handle changes to the transform
 * contents internally so that the program works properly regardless of
 * whether the program is receiving these events or not.
 * <P>
 * This event is generated by a ZTransformGroup node
 * when its transform is changed.
 * The event is passed to every <code>ZTransformListener</code>
 * or <code>ZTransformAdapter</code> object which registered to receive such
 * events using the transform's <code>addTransformListener</code> method.
 * (<code>ZTransformAdapter</code> objects implement the
 * <code>ZTransformListener</code> interface.) Each such listener object
 * gets this <code>ZTransformEvent</code> when the event occurs.
 *
 * <P>
 * <b>Warning:</b> Serialized and ZSerialized objects of this class will not be
 * compatible with future Jazz releases. The current serialization support is
 * appropriate for short term storage or RMI between applications running the
 * same version of Jazz. A future release of Jazz will provide support for long
 * term persistence.
 *
 * @see ZTransformAdapter
 * @see ZTransformListener
 * @author Ben Bederson
 */
public class ZTransformEvent extends AWTEvent implements ZEvent, Serializable {

    /**
     * The first number in the range of ids used for transform events.
     */
    public static final int TRANSFORM_FIRST       = 200;

    /**
     * The last number in the range of ids used for transform events.
     */
    public static final int TRANSFORM_LAST        = 200;

    /**
     * This event indicates that the transform changed.
     */
    public static final int TRANSFORM_CHANGED = TRANSFORM_FIRST;

    /**
     * The value of the transform before the transform was changed
     * for transform events (or null for other event types).
     */
    public AffineTransform transform = null;

    /**
     * Constructs a ZTransformEvent object.
     *
     * @param source    the ZTransform object that originated the event
     * @param id        an integer indicating the type of event
     * @param transform The original transform (for transform events)
     * @deprecated as of Jazz 1.1, use createTransfomrChangedEvent() instead.
     */
    public ZTransformEvent(ZTransformGroup source, int id, AffineTransform transform) {
        super(source, id);
        this.transform = transform;
    }

    protected ZTransformEvent(ZTransformGroup source, int id, AffineTransform transform, Object dummy) {
        super(source, id);
        this.transform = transform;
    }

    /**
     * Factory method to create a ZTransformEvent with a TRANSFORM_CHANGED ID.
     *
     * @param source    the ZTransform object that originated the event
     * @param transform The original transform (for transform events)
     */
    public static ZTransformEvent createTransformChangedEvent(ZTransformGroup source, AffineTransform transform) {
        return new ZTransformEvent(source, ZTransformEvent.TRANSFORM_CHANGED, transform, null);
    }

    /**
     * Returns the originator of the event.
     *
     * @return the ZTransformGroup object that originated the event
     */
    public ZTransformGroup getTransform() {
        return (ZTransformGroup)source;   // Cast is ok, checked in constructor
    }

    /**
     * For transform change events, this returns the value of the transform
     * before the transform was changed.
     *
     * @return the original transform value.
     */
    public AffineTransform getOrigTransform() {
        return transform;
    }

    /**
     * Calls appropriate method on the listener based on this events ID.
     */
    public void dispatchTo(Object listener) {
        ZTransformListener transformListener = (ZTransformListener) listener;
        switch (getID()) {
            case ZTransformEvent.TRANSFORM_CHANGED:
                transformListener.transformChanged(this);
                break;
            default:
                throw new RuntimeException("ZTransformEvent with bad ID");
        }
    }

    /**
     * Returns the ZTransformLister class.
     */
    public Class getListenerType() {
        return ZTransformListener.class;
    }

    /**
     * True if this event has been consumed.
     */
    public boolean isConsumed() {
        return super.isConsumed();
    }

    /**
     * Set the souce of this event. As the event is fired up the tree the source of the
     * event will keep changing to reflect the scenegraph object that is firing the event.
     */
    public void setSource(Object aSource) {
        source = aSource;
    }
}